/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.qna.client.api;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.osgi.service.event.Event;

import com.adobe.cq.social.commons.events.CommentEvent.CommentActions;
import com.adobe.cq.social.forum.client.api.Forum;
import com.adobe.cq.social.forum.client.api.Post;
import com.adobe.cq.social.qna.client.api.QnaEvent.QnaActions;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.core.SocialEvent;
import com.adobe.granite.activitystreams.ObjectTypes;
import com.adobe.granite.activitystreams.Verbs;

/**
 * Defines events that are triggered in a QnA Forum.
 */
public class QnaEvent extends SocialEvent<QnaEvent.QnaActions> {

    /**
     * The event topic suffix for QnA events.
     */
    public static final String QNA_TOPIC = "qna";

    private static final String QUESTION_PATH = "QUESTION_PATH";

    private static final String ANSWER_PATH = "ANSWER_PATH";

    private static final String QNA_FORUM_PATH = "QNA_FORUM_PATH";

    private static final long serialVersionUID = 1L;

    /**
     * List of available QnA actions that can trigger a QnA event.
     */
    public static enum QnaActions implements SocialEvent.SocialActions {
        QUESTION_CREATED, QUESTION_EDITED, QUESTION_DELETED, ANSWER_SELECTED, ANSWER_UNSELECTED, ANSWER_CREATED, ANSWER_EDITED, ANSWER_DELETED;

        private static final List<String> QNA_ACTIONS = new ArrayList<String>();
        static {
            for (QnaActions action : QnaActions.values()) {
                QNA_ACTIONS.add(QNA_TOPIC + "." + action.getVerb());
            }
        }

        public static List<String> getAvailableActions() {
            return QNA_ACTIONS;
        }

        @Override
        public String getVerb() {
            switch (this) {
                case QUESTION_CREATED:
                    return Verbs.POST;
                case QUESTION_EDITED:
                    return Verbs.UPDATE;
                case QUESTION_DELETED:
                    return Verbs.DELETE;
                case ANSWER_SELECTED:
                    return "select";
                case ANSWER_UNSELECTED:
                    return "unselect";
                case ANSWER_CREATED:
                    return Verbs.ADD;
                case ANSWER_EDITED:
                    return Verbs.UPDATE;
                case ANSWER_DELETED:
                    return Verbs.DELETE;
                default:
                    throw new IllegalArgumentException();
            }
        }
    }

    /**
     * @param path - the jcr path to either the question node or the answer node on which the event was triggered.
     * @param userId - the user id of the user who triggered the event.
     * @param action - the action that was taken to trigger the event.
     * @param additionalData - any additional data that is part of the event payload.
     */
    private QnaEvent(final String path, final String userId, final QnaActions action,
        final Map<String, Object> additionalData) {
        super(QNA_TOPIC, path, userId, action, additionalData);
    }

    /**
     * This constructor should be used to create a QnA Event given an OSGi Event.
     * @param event an OSGi event that is known to be a QnA event.
     */
    private QnaEvent(final Event event) {
        super(event);
    }

    private static QnaEvent buildQuestionEvent(final QnaPost post, final String userId, final QnaActions action) {
        final Map<String, Object> addData = new HashMap<String, Object>(2);
        addData.put(OBJECT, new BaseEventObject(getDisplayName(post, action), post.getResource().getPath(),
            ObjectTypes.ARTICLE));
        addData.put(TARGET,
            new BaseEventObject(getDisplayName(post.getParentComponent(), action), post.getSourceComponentId(),
                ObjectTypes.COLLECTION));
        return new QnaEvent(post.getResource().getPath(), userId, action, addData);
    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#ANSWER_UNSELECTED}.
     * @param post - the answer was selected.
     * @param userId - the user id of the user who selected the answer.
     * @return the event
     */
    public static QnaEvent answerUnselected(final QnaPost post, final String userId) {
        return buildQuestionEvent(post, userId, QnaActions.ANSWER_UNSELECTED);

    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#ANSWER_SELECTED}.
     * @param post - the answer was selected.
     * @param userId - the user id of the user who selected the answer.
     * @return the event
     */
    public static QnaEvent answerSelected(final QnaPost post, final String userId) {
        return buildQuestionEvent(post, userId, QnaActions.ANSWER_SELECTED);

    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#QUESTION_CREATED}.
     * @param post - the Qna that was created
     * @param userId - the user id of the user who created the question.
     * @return the event
     */
    public static QnaEvent questionCreated(final QnaPost post, final String userId) {
        return buildQuestionEvent(post, userId, QnaActions.QUESTION_CREATED);
    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#QUESTION_EDITED}.
     * @param post - the question that was edited
     * @param userId - the user id of the user who edited the question.
     * @return the event
     */
    public static QnaEvent questionEdited(final QnaPost post, final String userId) {
        return buildQuestionEvent(post, userId, QnaActions.QUESTION_EDITED);
    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#QUESTION_DELETED}.
     * @param post - the question that was deleted.
     * @param userId - the user id of the user who deleted the question.
     * @return the event
     */
    public static QnaEvent questionDeleted(final QnaPost post, final String userId) {
        return buildQuestionEvent(post, userId, QnaActions.QUESTION_DELETED);
    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#ANSWER_CREATED}.
     * @param post - the answer that was created.
     * @param userId - the user id for the user who created the answer.
     * @return the event
     */
    public static QnaEvent answerCreated(final QnaPost post, final String userId) {
        return buildAnswerEvent(post, userId, QnaActions.ANSWER_CREATED);
    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#ANSWER_DELETED}.
     * @param post - the answer that was deleted.
     * @param userId - the user id for the user who deleted the answer.
     * @return the event
     */
    public static QnaEvent answerDeleted(final QnaPost post, final String userId) {
        return buildAnswerEvent(post, userId, QnaActions.ANSWER_DELETED);
    }

    /**
     * Creates a QnAEvent that represents an action of {@link QnaActions#ANSWER_EDITED}.
     * @param post - the answer that was edited.
     * @param userId - the user id for the user who edited the answer.
     * @return the event
     */
    public static QnaEvent answerEdited(final QnaPost post, final String userId) {
        return buildAnswerEvent(post, userId, QnaActions.ANSWER_EDITED);
    }

    private static QnaEvent buildAnswerEvent(final QnaPost post, final String userId, final QnaActions action) {
        final Map<String, Object> addData = new HashMap<String, Object>(2);
        if (post.isTopic()) {
            // Question
            addData.put(OBJECT, new BaseEventObject(getDisplayName(post, action), post.getResource().getPath(),
                ObjectTypes.ARTICLE));
            addData.put(TARGET,
                new BaseEventObject(getDisplayName(post.getSourceComponent(), action), post.getSourceComponentId(),
                    ObjectTypes.COLLECTION));
        } else {
            // Answer
            addData.put(TARGET,
                new BaseEventObject(getDisplayName(post.getParentComponent(), action), post.getParentId(),
                    ObjectTypes.ARTICLE));
            addData.put(OBJECT, new BaseEventObject(getDisplayName(post, action), post.getResource().getPath(),
                ObjectTypes.COMMENT));
        }

        return new QnaEvent(post.getResource().getPath(), userId, action, addData);
    }

    private static String getDisplayName(final SocialComponent post, final QnaActions action) {
        if (post instanceof QnaPost) {
            final String name = (action.getVerb().equals(Verbs.DELETE)) ? "" : ((QnaPost) post).getSubject();
            if (name != null) {
                return name;
            } else {
                return ((QnaPost) post).isTopic() ? "a question" : "an answer";
            }
        } else {
            if (post instanceof Forum) {
                return ((Forum) post).getTitle();
            } else {
                return null;
            }
        }
    }
}