/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.reporting.analytics.api;

import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.community.api.CommunityContext;
import com.adobe.cq.social.reporting.analytics.AnalyticsConstants;
import com.adobe.cq.social.reporting.analytics.services.AnalyticsReportManagementService;
import com.adobe.cq.social.reporting.dv.api.AbstractDVLineChartComponent;
import com.adobe.cq.social.reporting.dv.model.api.LineChartModel;
import com.adobe.cq.social.scf.ClientUtilities;

public abstract class AbstractAnalyticsTrendReportComponent extends AbstractDVLineChartComponent {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractAnalyticsTrendReportComponent.class);

    /**
     * Service that handles Analytics mappings and queries.
     */
    protected AnalyticsReportManagementService analyticsReportService;

    /**
     * Community context for the Analytics report.
     */
    protected CommunityContext ctx;

    /**
     * Flag to track if this request initiated from the client (true) or from the server (false).
     */
    protected boolean isQuery = false;

    /**
     * @param resource
     * @param clientUtils
     * @param analyticsReportService - the service for running Analytics queries.
     * @param isQuery - for server-side rendering, we set this to false. Subsequent requests from client (AJAX) will
     *            set this to true.
     */
    public AbstractAnalyticsTrendReportComponent(Resource resource, ClientUtilities clientUtils,
        AnalyticsReportManagementService analyticsReportService, boolean isQuery) {
        super(resource, clientUtils);

        this.analyticsReportService = analyticsReportService;
        this.isQuery = isQuery;
    }

    /**
     * If has been correctly configured, runs an Analytics query and parses the response into DV-compatible format.
     */
    protected void querySiteCatalyst() {

        if (!isAnalyticsInitialized()) {
            return;
        }

        try {
            JSONObject reportDescription = prepareReport();

            if (reportDescription != null) {
                JSONObject result =
                    analyticsReportService.runAnalyticsReport(resource.getResourceResolver(), ctx,
                        AnalyticsConstants.COMMUNITIES_ANALYTICS_BASE_RESOURCE_TYPE, reportDescription);
                responseObject = parseResponse(result);
            }
        } catch (JSONException e) {
            LOGGER.error("Failure occurred while running analytics trend report: {}", e);
        }
    }

    /**
     * Returns true iff Analytics mappings have been configured.
     * @return
     */
    public boolean isAnalyticsInitialized() {
        if (!isQuery
                || !analyticsReportService.isAnalyticsConfigured(resource.getResourceResolver(), ctx,
                    AnalyticsConstants.COMMUNITIES_ANALYTICS_BASE_RESOURCE_TYPE)) {
            return false;
        }
        return true;
    }

    /**
     * Transforms the Analytics query response input into a format that DV client charting engine can consume. For
     * Analytics response format, see here:
     * https://marketing.adobe.com/developer/documentation/analytics-reporting-1-4
     * /r-reportresponse-1#reference_450F0CB0774E4716BE87F934C6C5BF95
     * @param input
     * @return DV-compatible response object
     * @throws JSONException
     */
    protected abstract LineChartModel parseResponse(JSONObject input) throws JSONException;

    /**
     * Constructs and returns the JSON input object for an Analytics query. For Analytics query input format, see
     * here: https://marketing.adobe.com/developer/documentation/analytics-reporting-1-4/r-queue
     * @return
     * @throws JSONException
     */
    protected abstract JSONObject prepareReport() throws JSONException;

    /**
     * Fetch the first value for a key in a multi-value property map.
     * @param map
     * @param key
     * @return
     */
    public static String getParameterValue(Map<String, String[]> map, String key) {
        if (map == null) {
            return null;
        }

        String[] values = map.get(key);
        return values == null ? null : values[0];
    }
}
