/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.review.client.api;

import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.jcr.RepositoryException;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.NonExistingResource;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.SyntheticResource;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.comments.api.AbstractComment;
import com.adobe.cq.social.commons.comments.api.CommentCollectionConfiguration;
import com.adobe.cq.social.commons.comments.listing.CommentSocialComponentListProviderManager;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.tally.TallyConstants;
import com.adobe.cq.social.tally.client.api.VotingSocialComponent;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.day.cq.wcm.api.designer.Style;

public class AbstractReview<C extends CommentCollectionConfiguration> extends AbstractComment<C> implements
    ReviewSocialComponent<C> {
    protected final Resource resource;
    protected final ResourceResolver resolver;
    protected final Map<String, String> ratingPaths;
    private List<String> allowedRatings;
    private Style style;

    /** Logger for this class. */
    private static final Logger log = LoggerFactory.getLogger(AbstractReview.class);

    /**
     * Construct a Review for the specified target resource.
     * @param resource the specified resource
     * @param clientUtils the client utilities instance
     * @param listProviderManager list manager to use for listing content
     * @throws RepositoryException if an error occurs
     */
    public AbstractReview(final Resource resource, final ClientUtilities clientUtils,
        final CommentSocialComponentListProviderManager listProviderManager) throws RepositoryException {
        this(resource, clientUtils, QueryRequestInfo.DEFAULT_QUERY_INFO_FACTORY.create(), listProviderManager);
    }

    /**
     * Constructor of a Review.
     * @param resource the specified {@link com.adobe.cq.social.commons.Comment}
     * @param clientUtils the client utilities instance
     * @param queryInfo the query info.
     * @param listProviderManager list manager to use for listing content
     * @throws RepositoryException if an error occurs
     */
    public AbstractReview(final Resource resource, final ClientUtilities clientUtils,
        final QueryRequestInfo queryInfo, final CommentSocialComponentListProviderManager listProviderManager)
        throws RepositoryException {
        super(resource, clientUtils, queryInfo, listProviderManager);
        this.resource = resource;
        this.resolver = resource.getResourceResolver();
        this.ratingPaths = getRatingPaths();
    }

    protected Map<String, String> getRatingPaths() {
        final Map<String, String> results = new LinkedHashMap<String, String>();
        final ValueMap values = this.resource.adaptTo(ValueMap.class);
        final String[] responsePaths = values.get(ReviewConstants.RATINGS_PROPERTY, new String[0]);
        for (int i = 0; i < responsePaths.length - 1; i = i + 2) {
            results.put(responsePaths[i], responsePaths[i + 1]);
        }
        return results;
    }

    @Override
    public Map<String, Long> getRatingResponses() throws ReviewException {
        final Map<String, Long> results = new LinkedHashMap<String, Long>();
        for (final Entry<String, String> entry : ratingPaths.entrySet()) {
            if (!getAllowedRatings().contains(entry.getKey())) {
                continue;
            }

            final String ratingPath = entry.getValue();
            final Resource r = this.resolver.resolve(ratingPath);
            final Long value = getValueFromResponseResource(r);
            results.put(entry.getKey(), value);
        }
        for (final String name : getAllowedRatings()) {
            if (!results.containsKey(name)) {
                results.put(name, null);
            }
        }
        return results;
    }

    private Long getValueFromResponseResource(final Resource resource) throws ReviewException {
        if (ResourceUtil.isNonExistingResource(resource)) {
            log.error("Response resource {} doesn not exist.", resource.getPath());
            return null;
        }
        try {
            final ValueMap properties = resource.adaptTo(ValueMap.class);
            String responseValue = properties.get(TallyConstants.RESPONSE_PROPERTY, String.class);
            if (TallyConstants.UNSET_RESPONSE_VALUE.equals(responseValue) || responseValue == null) {
                return null;
            }
            responseValue = responseValue.substring(0, responseValue.indexOf('.'));
            return Long.parseLong(responseValue);
        } catch (final NumberFormatException e) {
            throw logAndThrow(e);
        }
    }

    protected List<String> getAllowedRatings() {
        if (allowedRatings != null) {
            return allowedRatings;
        }
        final SocialComponent sourceComponent = this.getSourceComponent();
        final Resource reviewCollection;
        ValueMap values;
        if (sourceComponent != null) {
            reviewCollection = sourceComponent.getResource();
        } else {
            final String slingIncludedPath = this.getSourceComponentId();
            if (StringUtils.isEmpty(slingIncludedPath)) {
                log.error("Could not determine root component for review: {}", this.getId().toString());
                return allowedRatings;
            }
            reviewCollection = new NonExistingResource(resolver, slingIncludedPath);
        }
        values = reviewCollection.adaptTo(ValueMap.class);

        if (!ResourceUtil.isNonExistingResource(reviewCollection) && (values != null)) {
            final String[] ratings =
                values.get(ReviewConstants.ALLOWED_RATINGS_PROPERTY,
                    new String[]{ReviewConstants.PATH_OVERALL_RATING});
            allowedRatings = Arrays.asList(ratings);
        } else {
            values = clientUtils.getDesignProperties(reviewCollection, ReviewConstants.REVIEWS_RESOURCE_TYPE);
            if (values == null) {
                log.warn("Allowed ratings property is not set for Review {}, using default", this.resource.getPath());
                allowedRatings = Arrays.asList(ReviewConstants.PATH_OVERALL_RATING);
            } else {
                final String[] ratings =
                    values.get(ReviewConstants.ALLOWED_RATINGS_PROPERTY,
                        new String[]{ReviewConstants.PATH_OVERALL_RATING});
                allowedRatings = Arrays.asList(ratings);
            }

        }

        return allowedRatings;
    }

    @Override
    public boolean isCompositeRating() {
        if (getAllowedRatings() == null) {
            return false;
        }
        return getAllowedRatings().size() != 1;
    }

    @Override
    public VotingSocialComponent getVoting() {
        Resource voteResource = resolver.resolve(this.resource.getPath() + "/" + ReviewConstants.PATH_VOTING);
        if (ResourceUtil.isNonExistingResource(voteResource)) {
            // when the voting resource does not exist, create a synthetic resource to work with
            // SocialComponentFactory
            voteResource =
                new SyntheticResource(resolver, voteResource.getPath(), VotingSocialComponent.VOTING_RESOURCE_TYPE);
        }
        final SocialComponentFactoryManager scfMgr = clientUtils.getSocialComponentFactoryManager();
        if (scfMgr != null) {
            final SocialComponentFactory scf = scfMgr.getSocialComponentFactory(voteResource);
            if (scf != null) {
                return (VotingSocialComponent) scf.getSocialComponent(voteResource);
            }
        }
        return null;
    }

    private ReviewException logAndThrow(final Exception e) {
        log.error("Exception encounterred when trying to read response from node", e);
        return new ReviewException("Exception encounterred when trying to read response from node", e);
    }
}
