/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.review.client.api;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.SyntheticResource;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.commons.comments.listing.CommentSocialComponentListProviderManager;
import com.adobe.cq.social.review.client.endpoints.ReviewOperations;
import com.adobe.cq.social.review.client.impl.ReviewCollectionSocialComponentImpl;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.core.AbstractSocialComponentFactory;
import com.adobe.cq.social.ugc.api.UgcSearch;

/**
 * Social Comment Component Factory which allows the framework to create an {@link SocialComponent} for a Review
 * resource.
 */
@Component
@Service
public class ReviewCollectionSocialComponentFactory extends AbstractSocialComponentFactory implements
    SocialComponentFactory {

    @Reference
    private CommentSocialComponentListProviderManager listProviderManager;

    @Reference
    private UgcSearch search;
    /**
     * Logger.
     */
    private static final Logger LOG = LoggerFactory.getLogger(ReviewOperations.class);

    /**
     * Get the <code>SocialComponent</code> for the specified resource.
     * @param resource the <code>ReviewSystem</code> resource.
     * @return the target <code>SocialComponent</code>
     */
    @Override
    public SocialComponent getSocialComponent(final Resource resource) {
        return new ReviewCollectionSocialComponentImpl(resource, getClientUtilities(resource.getResourceResolver()),
            listProviderManager, search);
    }

    /**
     * Get the <code>SocialComponent</code> for the specified resource and request.
     * @param resource the <code>ReviewSystem</code> resource
     * @param request the client request
     * @return the target <code>SocialComponent</code>
     */
    @Override
    public SocialComponent getSocialComponent(final Resource resource, final SlingHttpServletRequest request) {
        if (resource.isResourceType(ReviewConstants.REVIEW_SUMMARY_RESOURCE_TYPE)) {
            final ValueMap componentConfigProperties;
            final ClientUtilities clientUtils = getClientUtilities(request);
            // Get the design if the component is a non existing resource.
            if (ResourceUtil.isNonExistingResource(resource)) {
                componentConfigProperties = clientUtils.getDesignProperties(resource, resource.getResourceType());
            } else {
                // This is the par component.
                componentConfigProperties = resource.getValueMap();
            }
            // Check the config for a path to use or just use the current path, this is how you get nonexisting
            // resources
            // working. The NER path becomes the location of the reviews component.
            final String targetReviewsPath =
                componentConfigProperties.get(ReviewConstants.REVIEW_PATH_PROPERTY, resource.getPath());

            Resource reviewResource = resource.getResourceResolver().resolve(targetReviewsPath);
            if (ResourceUtil.isNonExistingResource(reviewResource)) {
                // If the target review resource is non existing we need to reconstruct a synthetic resource pointing
                // at the
                // target reviews path, which itself could be nonexisting!
                final Resource targetResource = resource.getResourceResolver().resolve(targetReviewsPath);
                // Also force the resource type of the resource back to the one that we came in with ( in case this is
                // an
                // Extension )
                reviewResource =
                    new SyntheticResource(resource.getResourceResolver(), targetResource.getResourceMetadata(),
                        resource.getResourceType());
            }

            final ReviewCollectionSocialComponentImpl rc =
                new ReviewCollectionSocialComponentImpl(reviewResource, getClientUtilities(request),
                    getQueryRequestInfo(request), listProviderManager, search);
            final boolean includeHistogram =
                componentConfigProperties.get(ReviewConstants.INCLUDE_HISTOGRAM_PROPERTY, false);
            rc.setIncludeHistogram(includeHistogram);
            return rc;
        }
        return new ReviewCollectionSocialComponentImpl(resource, getClientUtilities(request),
            getQueryRequestInfo(request), listProviderManager, search);
    }

    @Override
    public SocialComponent getSocialComponent(final Resource resource, final ClientUtilities clientUtils,
        final QueryRequestInfo queryInfo) {
        return new ReviewCollectionSocialComponentImpl(resource, clientUtils, queryInfo, listProviderManager, search);
    }

    /**
     * Get the supported resource type.
     * @return <code>ReviewSystem.RESOURCE_TYPE</code>
     */
    @Override
    public String getSupportedResourceType() {
        return ReviewConstants.REVIEWS_RESOURCE_TYPE;
    }

}
