/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.scf;

import java.security.acl.Group;
import java.util.List;
import java.util.Set;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.jcr.RepositoryException;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.request.RequestParameter;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;

import com.adobe.cq.social.srp.config.SocialResourceConfiguration;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.adobe.cq.social.ugcbase.core.attachments.FileDataSource;
import com.adobe.cq.social.ugcbase.core.attachments.FileUploadSizeLimit;
import com.adobe.granite.security.user.UserProperties;
import com.adobe.granite.translation.core.MachineTranslationCloudConfig;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.webservicesupport.Configuration;

/**
 * Provides helper methods that aid in performing some request specific functions like XSS, URL generations, etc.
 */
public interface ClientUtilities {

    /** JSON extension. */
    String JSON_EXTENSION = ".json";
    /** GET Selector. */
    String SOCO_SELECTOR = ".social";

    /** The query selector. */
    String QUERY_SELECTOR = ".query";

    String USER_ROOTPATH = "/social/authors/";

    /**
     * Computes the externalized URL to the resource at the given path.
     * @param path path to the resource
     * @return an externalized url to the resource
     */
    String externalLink(final String path);

    /**
     * Computes the externalized URL to the resource at the given path.
     * @param path path to resource
     * @param needExtension boolean to indicate whether the .social.json extension needs to be appended to the URL
     * @return an externalized url to the resource with or without .social.json suffix.
     */
    String externalLink(final String path, final Boolean needExtension);

    /**
     * Get the current extension of the request. Useful when building URLS based on the requested content type. If the
     * constructor that takes in a ResourceResolver was used then this information is not available and null is
     * returned.
     * @return the extension without a leading dot
     */
    String getRequestExtension();

    /**
     * Calls {@Link XSSAPI} filterHTML on the provided String. If no reference to XSSAPI is available the
     * String is returned as is.
     * @param valueToBeFiltered The String to be filtered.
     * @return The filtered String.
     */
    String filterHTML(final String valueToBeFiltered);

    /**
     * Get the {@link SocialComponentFactoryManager} service.
     * @return the SocialComponentFactoryManager service.
     */
    SocialComponentFactoryManager getSocialComponentFactoryManager();

    /**
     * Gets the {@link SocialUtils} service.
     * @return a SocialUtils implementation
     */
    SocialUtils getSocialUtils();

    /**
     * Gets the authorized user ID. If the constructor that takes a ResourceResolver was used then this information is
     * not available and null will be returned.
     * @return The authorized user ID, or null.
     */
    String getAuthorizedUserId();

    /**
     * Helper method to check whether the user is anonymous (not logged in) or not. If the constructor that takes a
     * ResourceResolver was used then this will always return true.
     * @return true if the authorized userID is null or anonymous
     */
    boolean userIsAnonymous();

    /**
     * Get the original request.
     * @return the {@link SlingHttpServletRequest} request
     */
    SlingHttpServletRequest getRequest();

    /**
     * Helper method to get the configured social profile path for the page on which the resource is contained. If the
     * constructor that takes a ResourceResolver is used this information is not available and null will be returned.
     * @return The configured social profile (public) path if one exists, or null.
     */
    String getSocialProfilePath();

    /**
     * Returns the page that contains the resource at the given path if path is absolute. If path is relative, the
     * request resources path is pre-pended to the path in order to construct an absolute path. If the constructor
     * that takes a ResourceResolver was used then this will always return null. request path to construct an absolute
     * path.
     * @param path the path to the resource for which the containing page is to be found
     * @return the page that contains the resource if one is found
     */
    Page getContainingPage(String path);

    /**
     * Returns the {@Link Page} from the specified path.
     * @param pathToPage path to a page
     * @return {@link Page} object from the page path, or null.
     */
    Page getPage(String pathToPage);

    /**
     * Get the {@Link User} for the specified user ID.
     * @param userId The user ID of the user.
     * @param resolver The resolver to use.
     * @return A {@link User} representing the user id.
     */
    User getUser(String userId, ResourceResolver resolver);

    /**
     * Get the {@link User} for the specified user id with the specified resource type.
     * @param userId The user id
     * @param resolver The resolver to use
     * @param resourceType the resource type
     * @return the user
     */
    User getUser(String userId, ResourceResolver resolver, String resourceType);

    /**
     * Get CQ userId from the id retrieved from the {@link getUser} request
     */
    String getUserId(final String userId);

    /**
     * Gets an active storage cloud service configuration based on the requested resource.
     * @param resource The resource to get the active storage cloud service for.
     * @return a Cloud {@Link Configuration} for the specified resource, or null if none exists.
     */
    Configuration getStorageCloudConfig(Resource resource);

    /**
     * Gets an active storage configuration based on the requested resource.
     * @param resource The resource to get the active storage configuration for.
     * @return a configuration for the specified resource, or null if none exists.
     */
    SocialResourceConfiguration getStorageConfig(Resource resource);

    /**
     * Gets an Machine Translation config for the requested resource.
     * @param resource The resource to get the Machine Translation config for.
     * @return A {@Link MachineTranslationCloudConfig} for the requested resource, or null if none exists.
     */
    MachineTranslationCloudConfig getMTCloudConfig(Resource resource);

    /**
     * Checks if Translation Service is configured for the requested resource.
     * @param resource The resource to get the Machine Translation config for.
     * @return true, if Translation service is configured correctly, or false otherwise
     */
    boolean isTranslationServiceConfigured(Resource resource);

    /**
     * Checks if Analytics Service is configured for the requested resource.
     * @param resource The resource to get the Analytics config for.
     * @return true, if Analytics service is configured correctly, or false otherwise
     */
    boolean isAnalyticsServiceConfigured(Resource resource);

    /**
     * Utility to filter the attachments in the current request against the specified white list, black list, and
     * size.
     * @param attachmentSizeLimit Size limit for attachments.
     * @param whiteList {@link java.util.List} of allowed file types.
     * @param blackList File types that are not allowed.
     * @return {@link java.util.List} of filtered attachments.
     */
    @Nonnull
    List<FileDataSource> getAttachmentsFromRequest(final long attachmentSizeLimit, final List<String> whiteList,
        final List<String> blackList);

    /**
     * Utility to check attached files against the specified white list and black list.
     * @param fileRequestParameters Request parameters of the type file.
     * @param attachmentSizeLimit Size limit for attachments.
     * @param whitelist {@link java.util.List} of allowed file types.
     * @param blacklist File types that are not allowed.
     * @return {@link java.util.List} of filtered attachments.
     */
    @Nonnull
    List<FileDataSource> getAttachmentsFromRequest(final RequestParameter[] fileRequestParameters,
        final long attachmentSizeLimit, final List<String> whitelist, final List<String> blacklist);

    /**
     * Utility to check attached files against the specified white list and black list.
     * @param fileRequestParameters Request parameters of the type file.
     * @param fileUploadSizeLimit {@link com.adobe.cq.social.scf.core.FileUploadSizeLimit} of the file.
     * @param whitelist List of allowed file types.
     * @param blackList File types that are not allowed.
     * @return {@link java.util.List} of filtered attachments.
     */
    @Nonnull
    List<FileDataSource> getAttachmentsFromRequest(final RequestParameter[] fileRequestParameters,
        final FileUploadSizeLimit fileUploadSizeLimit, final List<String> whitelist, final List<String> blackList);

    /**
     * Utility to check attached files against the specified white list and black list.
     * @param attachmentList Unfiltered list of attachments.
     * @param fileUploadSizeLimit {@link com.adobe.cq.social.scf.core.FileUploadSizeLimit} of the file.
     * @param whitelist {@link java.util.List} of allowed file types.
     * @param blacklist File types that are not allowed.
     * @return List of filtered {@link com.adobe.cq.social.scf.core.FileDataSource}
     */
    @Nonnull
    Iterable<FileDataSource> getAttachmentsFromDataSources(final Iterable<FileDataSource> attachmentList,
        final FileUploadSizeLimit fileUploadSizeLimit, final Set<String> whitelist, final List<String> blacklist);

    /**
     * Utility to check a file against the specified white list, black list, and size restraints.
     * @param attachmentDataSource {@link com.adobe.cq.social.scf.core.FileDataSource} of the attachment.
     * @param fileUploadSizeLimit {@link com.adobe.cq.social.scf.core.FileUploadSizeLimit} of the file.
     * @param whitelist {@link java.util.Set} of allowed file types.
     * @param blacklist File types that are not allowed.
     * @return {@link com.adobe.cq.social.scf.core.FileDataSource} for the file. Or null if the file is not allowed.
     */
    @CheckForNull
    FileDataSource getAttachmentFromDataSource(final FileDataSource attachmentDataSource,
        final FileUploadSizeLimit fileUploadSizeLimit, final Set<String> whitelist, final List<String> blacklist);

    /**
     * This Will attempt to get the design based on the component resource. If the component resource is a non
     * existing resource, it will attempt to get the design from there. If the NER is not in the content tree it will
     * create a path based on the last include location + the last token in the NER's path.
     * @param componentResource what component resource to try and get the design properties from
     * @param restrictToType only allow this method to return if the requested type hint isA certain type
     * @return a value map of the component's design properties
     */
    ValueMap getDesignProperties(final Resource componentResource, final String restrictToType);

    /**
     * Gets the included component when Sling performed the include on the server.
     * @return the resource that was included to produce this component.
     */
    Resource getIncludedResource();

    /**
     * Get the {@link Group} for the specified user id with the specified resource type.
     * @param id
     * @param resolver
     * @param resourceType
     * @return
     */
    SocialGroup getGroup(String id, ResourceResolver resolver, String resourceType);

    /**
     * Get the {@link UserProperties} for the authenticated user.
     * @return UserProperties
     * @throws RepositoryException if a repository problem is encountered.
     */
    UserProperties getAuthorizedUserProfileProperties() throws RepositoryException;

    /**
     * Get the {@link UserProperties} for the specified user
     * @param authorizableId the user whose properties to get.
     * @param relPath the relative path within the user to find the properties, usually 'profile' or 'preferences'.
     * @return UserProperties of the specified user from the specified path
     * @throws RepositoryException if a repository problem is encountered.
     */
    UserProperties getUserProperties(String authorizableId, String relPath) throws RepositoryException;

    /**
     * This method maps the given <code>resource</code> to its equivalent resource path based on applicable global
     * shared storage cloud config settings. This supports language specific pages hosting Communities components
     * which all store their content in a language neutral location in the SRP.
     * @param resource The resource to transpose.
     * @return string representing the global shared storage resource path.
     */
    String getCommonStorePath(Resource resource);
}
