/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.scf;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.Resource;

import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.CollectionPagination;
import com.adobe.cq.social.scf.SocialCollectionComponent;
import com.day.cq.wcm.api.Page;
import com.day.cq.commons.LanguageUtil;

/**
 * Shows information about the collection's pages.
 * @author mcweeney
 */
public class PageInfo {

    private final String urlPattern;
    private long totalItems = -1;
    private final CollectionPagination pagination;
    private final String params;
    private final String basePageURL;
    private final SocialCollectionComponent component;
    private final Resource includedResource;

    public PageInfo(final SocialCollectionComponent component, final ClientUtilities clientUtils,
        final CollectionPagination pagination) {
        this(component, clientUtils, pagination, null);
    }

    private String getRefererPath(final ClientUtilities clientUtils, final String path) {
        // Referer path:
        // from where the user made the request. In this case
        // http://localhost:4503/content/community-components/en/forum.topic.10.10.html/csc1-you_are_welocomeher (As
        // is in English page)

        // Path:
        // Where the forum topic was created. In this case
        // /content/community-components/de/forum ( As topic was created in German )

        // So, we replace the commentPostPageLanguage with refererLanguage to ensure that user doesn,t loose the
        // context.

        final String refererPath = clientUtils.getRequest().getHeader("referer");
        if (refererPath != null) {
            final StringBuilder refererLanguagePath;
            String refererLanguage = StringUtils.substringAfterLast(LanguageUtil.getLanguageRoot(refererPath), "/");
            String commentPostPageLanguage = StringUtils.substringAfterLast(LanguageUtil.getLanguageRoot(path), "/");
            if (refererLanguage != null && commentPostPageLanguage != null
                    && !refererLanguage.equals(commentPostPageLanguage)) {
                String leftPath = StringUtils.substringBeforeLast(LanguageUtil.getLanguageRoot(path), "/");
                String rightPath = StringUtils.substringAfterLast(path, "/");
                if (leftPath != null && rightPath != null) {
                    refererLanguagePath = new StringBuilder();
                    refererLanguagePath.append(leftPath).append("/");
                    refererLanguagePath.append(refererLanguage).append("/");
                    refererLanguagePath.append(rightPath);
                    return refererLanguagePath.toString();
                }
            }
        }
        // Return the path, if laguage are same or null
        return path;
    }

    public PageInfo(final SocialCollectionComponent component, final ClientUtilities clientUtils,
        final CollectionPagination pagination, final String params) {
        this.pagination = pagination;
        final Page page = clientUtils.getContainingPage(component.getId().toString());
        if (page != null && !StringUtils.isEmpty(pagination.getSelectedIndex())) {
            basePageURL =
                clientUtils.externalLink(
                    getRefererPath(clientUtils, page.getPath()) + ".index." + pagination.getSelectedIndex(), false);
        } else if (page != null) {
            basePageURL = clientUtils.externalLink(getRefererPath(clientUtils, page.getPath()), false);
        } else {
            basePageURL = null;
        }
        urlPattern =
            clientUtils.externalLink(component.getId() + ClientUtilities.SOCO_SELECTOR, false) + "."
                    + "${startIndex}." + ((Long) this.getPageSize()).toString() + "."
                    + clientUtils.getRequestExtension();
        this.params = params;
        this.component = component;
        this.includedResource = clientUtils.getIncludedResource();
    }

    /**
     * Gets the url pattern for fetching other pages with two tokens ${startIndex} and ${size} which determine where
     * to start the next page from and how many entries to fetch.
     * @return a string with the format http://host:port/pathtocomponent.${startIndex}.${size}.selectors.format
     */
    public String getURLPattern() {
        return urlPattern;
    }

    /**
     * Gets the current page number.
     * @return the current page number
     */
    public long getSelectedPage() {
        final Double pagesSeen = Math.ceil(pagination.getOffset() / Math.abs(this.getPageSize()));
        if (this.pagination.getSize() == 0) {
            return pagesSeen.longValue();
        }
        return ((Double) (pagesSeen + 1.0)).longValue();
    }

    /**
     * Gets the total number of pages in this component.
     * @return the total number of pages
     */
    public long getTotalPages() {
        return ((Double) Math.ceil(getTotalItems() * 1.0 / Math.abs(this.getPageSize()))).longValue();
    }

    /**
     * Gets the current configuration for the size of a page.
     * @return the size of a page in items
     */
    public long getPageSize() {
        return pagination.getSize() > 0 ? pagination.getSize() : pagination.getPageSizeConfig();
    }

    /**
     * The current offset into the index that is being used for the pages.
     * @return the current index
     */
    public long getCurrentIndex() {
        return pagination.getOffset();
    }

    /**
     * A simple function for getting the next page in the results. If it is the last page it points to this page.
     * @return a url for the next page
     */
    public String getNextPageURL() {

        String nextPage = StringUtils.replace(getURLPattern(), "${startIndex}", ((Long) getNextOffset()).toString());
        if (params != null) {
            nextPage += "?" + params;
        }
        return nextPage;
    }

    /**
     * A simple function for getting the next page in the results.
     * @return a url for the previous page
     */
    public String getPreviousPageURL() {
        String nextPage =
            StringUtils.replace(getURLPattern(), "${startIndex}", ((Long) getPreviousOffset()).toString());
        if (params != null) {
            nextPage += "?" + params;
        }
        return nextPage;
    }

    public String getNextSuffix() {
        return getNextOffset() + "." + this.getPageSize();
    }

    public String getPreviousSuffix() {
        return getPreviousOffset() + "." + this.getPageSize();
    }

    /**
     * The current offset into the index that is being used for the pages.
     * @return the current index
     */
    public String getSelectedIndex() {
        return pagination.getSelectedIndex();
    }

    /**
     * Returns the base page URL based on the current social component.
     * @return a string that points to the path to the page that contains the component.
     */
    public String getBasePageURL() {
        return this.basePageURL;
    }

    public String getIncludedPath() {
        if (this.includedResource != null) {
            return this.includedResource.getPath();
        }
        return null;
    }

    private long getTotalItems() {
        if (totalItems < 0) {
            this.totalItems = component.getTotalSize();
        }
        return this.totalItems;
    }

    private long getNextOffset() {
        return getTotalItems() < (Math.abs(pagination.getSize()) + pagination.getOffset()) ? getTotalItems() : Math
            .abs(pagination.getSize()) + pagination.getOffset();
    }

    private long getPreviousOffset() {
        return 0 >= (pagination.getOffset() - Math.abs(pagination.getSize())) ? 0 : (pagination.getOffset() - Math
            .abs(pagination.getSize()));

    }
}
