/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.scf.core;

import javax.mail.MethodNotSupportedException;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.commons.osgi.OsgiUtil;
import org.osgi.service.component.ComponentContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.ClientUtilityFactory;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.adobe.granite.xss.XSSAPI;

/**
 * Abstract base class for a social component factory. Makes XSSAPI and {@link ClientUtilities} available.
 */
@Component(metatype = false, componentAbstract = true)
public abstract class AbstractSocialComponentFactory implements SocialComponentFactory {
    /** Logger for this class. */
    private static final Logger LOG = LoggerFactory.getLogger(AbstractSocialComponentFactory.class);

    /** Reference to <code>XSSAPI</code>. */
    @Reference(cardinality = ReferenceCardinality.MANDATORY_UNARY, policy = ReferencePolicy.STATIC)
    private XSSAPI xss;

    @Reference
    private SocialUtils socialUtils;

    /** Priority level of the component */
    @Property(name = "priority", intValue = 0)
    protected int priority;

    @Reference
    ClientUtilityFactory clientUtilFactory;

    /**
     * Get an instance of the clientUtilities for the specified request.
     * @param request a specific SlingHttpServletRequest for which ClientUtilities instance is created.
     * @return an instance of ClientUtilities
     * @throws MethodNotSupportedException
     */
    protected ClientUtilities getClientUtilities(final SlingHttpServletRequest request) {
        return clientUtilFactory.getClientUtilities(xss, request, socialUtils);
    }

    protected ClientUtilities getClientUtilities(final ResourceResolver resourceResolver) {
        return clientUtilFactory.getClientUtilities(this.xss, resourceResolver, socialUtils);
    }

    /**
     * Get priority of this factory. This priority determines which factory is returned for a specific resource type.
     * Override this method to return a higher number to make this factory override existing factories. Default value
     * is 0.
     * @return priority of this factory
     */
    @Override
    public int getPriority() {
        return priority;
    }

    /**
     * Get the XSS Protection Service.
     * @return an instance of the XSSAPI service
     */
    protected XSSAPI getXssService() {
        return xss;
    }

    /**
     * Get the SocialUtils service for handling alternate storage.
     * @return an instance of SocialUtils
     */
    protected SocialUtils getSocialUtils() {
        return socialUtils;
    }

    /**
     * Activate this component. Open the session and register event listeners.
     * @param context The component context
     */
    protected void activate(final ComponentContext context) {
        priority = OsgiUtil.toInteger(context.getProperties().get("priority"), 0);
    }

    /**
     * Get the {@link QueryRequestInfo} from the specified request.
     * @param request
     * @return the QueryRequestInfo
     */
    protected QueryRequestInfo getQueryRequestInfo(final SlingHttpServletRequest request) {
        if (request != null) {
            return new BaseQueryRequestInfo(request);
        } else {
            return BaseQueryRequestInfo.DEFAULT_QUERY_REQUEST;
        }
    }

}
