/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.scf.core;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.request.RequestPathInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.scf.CollectionPagination;
import com.adobe.cq.social.scf.QueryRequestInfo;
import com.adobe.cq.social.scf.SocialComponent;

/**
 * Default implementation of the QueryRequestInfo.
 */
public class BaseQueryRequestInfo implements QueryRequestInfo {
    /** Logger for this class. */
    private static final Logger LOG = LoggerFactory.getLogger(BaseQueryRequestInfo.class);

    private static String PN_QUERY_FILTER = "filter";

    private final boolean isQueryRequest;
    private final Map<String, String[]> predicates;
    private CollectionPagination pagination;
    private CollectionSortedOrder order = CollectionSortedOrder.DEFAULT_ORDER;
    private List<String> sortBy;

    public static QueryRequestInfo DEFAULT_QUERY_REQUEST = new BaseQueryRequestInfo();

    private boolean isTranslateRequest;

    /**
     * Private constructor, creating a default Query Request Info.
     */
    private BaseQueryRequestInfo() {
        this.pagination = CollectionPagination.DEFAULT_PAGINATION;
        this.isQueryRequest = false;
        this.predicates = Collections.emptyMap();
    }

    /**
     * Constructor which parses the info from the specified {@link SlingHttpServletRequest}.
     * @param request the http servlet request.
     */
    public BaseQueryRequestInfo(final SlingHttpServletRequest request) {
        this(request, true);
    }

    /**
     * Constructor which parses the info from the specified {@link SlingHttpServletRequest}.
     * @param request the http servlet request.
     * @param isFilterQuery true, if a query in the request is defined by the 'filter' parameter, false otherwise
     */
    @SuppressWarnings("unchecked")
    public BaseQueryRequestInfo(final SlingHttpServletRequest request, final boolean isFilterQuery) {
        this.pagination = getPaginationFromRequest(request);
        this.isQueryRequest = isQueryRequest(request, isFilterQuery);
        this.predicates = (isQueryRequest) ? request.getParameterMap() : Collections.emptyMap();
        this.isTranslateRequest = isTranslateRequest(request);
    }

    /**
     * Constructor.
     * @param isQueryRequest
     * @param predicates
     * @param pagination
     */
    public BaseQueryRequestInfo(final boolean isQueryRequest, final Map<String, String[]> predicates,
        final CollectionPagination pagination) {
        this.isQueryRequest = isQueryRequest;
        this.predicates = predicates;
        this.pagination = pagination;
    }

    /**
     * Constructor.
     * @param isQueryRequest
     * @param predicates
     * @param pagination
     * @param isTranslateRequest
     */
    public BaseQueryRequestInfo(final boolean isQueryRequest, final Map<String, String[]> predicates,
        final CollectionPagination pagination, final boolean isTranslateRequest) {
        this.isQueryRequest = isQueryRequest;
        this.predicates = predicates;
        this.pagination = pagination;
        this.isTranslateRequest = isTranslateRequest;
    }

    /**
     * Constructor.
     * @param isQueryRequest
     * @param predicates
     * @param pagination
     */
    public BaseQueryRequestInfo(final boolean isQueryRequest, final Map<String, String[]> predicates,
        final CollectionPagination pagination, final CollectionSortedOrder order) {
        this.isQueryRequest = isQueryRequest;
        this.predicates = predicates;
        this.pagination = pagination;
        this.order = order;
    }

    /**
     * Constructor.
     * @param isQueryRequest
     * @param predicates
     * @param pagination
     */
    public BaseQueryRequestInfo(final boolean isQueryRequest, final Map<String, String[]> predicates,
        final CollectionPagination pagination, final CollectionSortedOrder order, final List<String> sortBy) {
        this.isQueryRequest = isQueryRequest;
        this.predicates = predicates;
        this.pagination = pagination;
        this.order = order;
        this.sortBy = sortBy;
    }

    /**
     * Parse the request and produce the <code>CollectionPagination</code>. This method assumes that the offset and
     * length selector is the last 2 selectors.
     * @param request
     * @return the specified pagination in the request; if the specified request is null, or there is no pagination
     *         selector in the request, the method returns the default
     *         <code>CollectionPagination.DEFAULT_PAGINATION</code>
     */
    protected CollectionPagination getPaginationFromRequest(final SlingHttpServletRequest request) {
        if (request != null) {
            // Pull out the "social" selector if it is there when we go to parse the selectors for our index
            // information.
            final String[] selectors =
                ArrayUtils.removeElement(request.getRequestPathInfo().getSelectors(),
                    SocialComponent.DEFAULT_SELECTOR);
            if (selectors != null && selectors.length >= 2) {
                try {
                    final int maxIndex = selectors.length - 1;  // zero base
                    final int offset = Integer.parseInt(selectors[maxIndex - 1]);
                    int size = Integer.parseInt(selectors[maxIndex]);
                    // Reverse order if the size is negative.
                    if (size < 0) {
                        size = Math.abs(size);
                        this.order = CollectionSortedOrder.REVERSED_ORDER;
                    }
                    String selectedIndex = null;
                    if (StringUtils.equals(selectors[0], "index")) {
                        selectedIndex = selectors[1];
                    }
                    final CollectionPagination pagination =
                        new CollectionPagination(offset, size, CollectionPagination.DEFAULT_EMBED_LEVEL,
                            selectedIndex);
                    return pagination;
                } catch (final NumberFormatException e) {
                    LOG.error("Bad pagination request to " + request.getPathInfo(), e);
                }
            }
        }
        return CollectionPagination.DEFAULT_PAGINATION;
    }

    /**
     * Check whether the specified request contains queryable parameters or not.
     * @param request the specified request.
     * @param isFilterQuery if we need to check for the presence of the filter param or not
     * @return
     */
    private boolean isQueryRequest(final SlingHttpServletRequest request, boolean isFilterQuery) {
        final RequestPathInfo info = request.getRequestPathInfo();
        final String selector = info.getSelectorString();
        boolean query = (selector != null && selector.contains(SocialComponent.QUERY_SELECTOR));
        if (!query) {
            query =
                (isFilterQuery) ? "GET".equalsIgnoreCase(request.getMethod())
                        && request.getParameterMap().containsKey(PN_QUERY_FILTER) : "GET".equalsIgnoreCase(request
                    .getMethod()) && !request.getParameterMap().isEmpty();
        }
        return query;
    }

    /**
     * Check whether the specified request contains translate selector or not.
     * @param request the specified request.
     * @return
     */
    private boolean isTranslateRequest(final SlingHttpServletRequest request) {
        final RequestPathInfo info = request.getRequestPathInfo();
        final String selector = info.getSelectorString();
        boolean isTranslate = (selector != null && selector.contains(SocialComponent.TRANSLATE_SELECTOR));

        return isTranslate;
    }

    @Override
    public boolean isQuery() {
        return isQueryRequest;
    }

    @Override
    public Map<String, String[]> getPredicates() {
        return predicates;
    }

    @Override
    public CollectionPagination getPagination() {
        return pagination;
    }

    @Override
    public void setPagination(final CollectionPagination pagination) {
        this.pagination = pagination;
    }

    @Override
    public void setSortBy(final List<String> sortBy) {
        this.sortBy = sortBy;
    }

    @Override
    public List<String> getSortBy() {
        return this.sortBy;
    }

    @Override
    public void setSortOrder(final CollectionSortedOrder order) {
        this.order = order;
    }

    @Override
    public CollectionSortedOrder getSortOrder() {
        return this.order;
    }

    @Override
    public boolean isTranslationRequest() {
        return isTranslateRequest;
    }

    @Override
    public void setTranslationRequest(boolean isTranslationRequest) {
        this.isTranslateRequest = isTranslationRequest;
    }
}
