/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.scf.core;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.InvalidPropertiesFormatException;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.JsonException;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.day.cq.wcm.api.Page;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.SerializableString;
import com.fasterxml.jackson.core.io.CharacterEscapes;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;

/**
 * Default SocialComponent implementation for all Resource Types. This class should be extended to implement
 * SocialComponent for other Resource Types.
 */
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.PUBLIC_ONLY, getterVisibility = Visibility.PUBLIC_ONLY,
        isGetterVisibility = Visibility.PUBLIC_ONLY, setterVisibility = Visibility.NONE)
@JsonInclude(Include.NON_EMPTY)
public class BaseSocialComponent implements SocialComponent {

    // Has to be this way because this is API and the Properties member is protected.
    private class LazyProperties extends Properties {

        boolean needPropertyValues = true;

        private synchronized void initPropertyVaues() {
            if (!needPropertyValues) {
                return;
            }
            needPropertyValues = false;
            final BaseSocialComponent thiz = BaseSocialComponent.this;
            final ValueMap propMap = thiz.resource.adaptTo(ValueMap.class);
            if (propMap != null) {
                initIgnoredPropertiesList();
                for (final String key : propMap.keySet()) {
                    if (!thiz.isKeyIgnored(key)) {
                        final Object value = propMap.get(key);
                        if (value == null) {
                            this.setProperty(key, "");
                        } else {
                            if (value.getClass().isArray()) {
                                this.put(key, value);
                            } else {
                                this.setProperty(key, value.toString());
                            }
                        }
                    }
                }
            }
        }

        @Override
        public synchronized Object setProperty(final String key, final String value) {
            initPropertyVaues();
            return super.setProperty(key, value);
        }

        @Override
        public synchronized void load(final Reader reader) throws IOException {
            initPropertyVaues();
            super.load(reader);
        }

        @Override
        public synchronized void load(final InputStream inStream) throws IOException {
            initPropertyVaues();
            super.load(inStream);
        }

        @Override
        public void save(final OutputStream out, final String comments) {
            initPropertyVaues();
            super.save(out, comments);
        }

        @Override
        public void store(final Writer writer, final String comments) throws IOException {
            initPropertyVaues();
            super.store(writer, comments);
        }

        @Override
        public void store(final OutputStream out, final String comments) throws IOException {
            initPropertyVaues();
            super.store(out, comments);
        }

        @Override
        public synchronized void loadFromXML(final InputStream in) throws IOException,
            InvalidPropertiesFormatException {
            initPropertyVaues();
            super.loadFromXML(in);
        }

        @Override
        public void storeToXML(final OutputStream os, final String comment) throws IOException {
            initPropertyVaues();
            super.storeToXML(os, comment);
        }

        @Override
        public void storeToXML(final OutputStream os, final String comment, final String encoding) throws IOException {
            initPropertyVaues();
            super.storeToXML(os, comment, encoding);
        }

        @Override
        public String getProperty(final String key) {
            initPropertyVaues();
            return super.getProperty(key);
        }

        @Override
        public String getProperty(final String key, final String defaultValue) {
            initPropertyVaues();
            return super.getProperty(key, defaultValue);
        }

        @Override
        public Enumeration<?> propertyNames() {
            initPropertyVaues();
            return super.propertyNames();
        }

        @Override
        public Set<String> stringPropertyNames() {
            initPropertyVaues();
            return super.stringPropertyNames();
        }

        @Override
        public void list(final PrintStream out) {
            initPropertyVaues();
            super.list(out);
        }

        @Override
        public void list(final PrintWriter out) {
            initPropertyVaues();
            super.list(out);
        }

        @Override
        public synchronized int size() {
            initPropertyVaues();
            return super.size();
        }

        @Override
        public synchronized boolean isEmpty() {
            initPropertyVaues();
            return super.isEmpty();
        }

        @Override
        public synchronized Enumeration<Object> keys() {
            initPropertyVaues();
            return super.keys();
        }

        @Override
        public synchronized Enumeration<Object> elements() {
            initPropertyVaues();
            return super.elements();
        }

        @Override
        public synchronized boolean contains(final Object value) {
            initPropertyVaues();
            return super.contains(value);
        }

        @Override
        public boolean containsValue(final Object value) {
            initPropertyVaues();
            return super.containsValue(value);
        }

        @Override
        public synchronized boolean containsKey(final Object key) {
            initPropertyVaues();
            return super.containsKey(key);
        }

        @Override
        public synchronized Object get(final Object key) {
            initPropertyVaues();
            return super.get(key);
        }

        @Override
        public synchronized Object put(final Object key, final Object value) {
            initPropertyVaues();
            return super.put(key, value);
        }

        @Override
        public synchronized Object remove(final Object key) {
            initPropertyVaues();
            return super.remove(key);
        }

        @Override
        public synchronized void putAll(final Map<? extends Object, ? extends Object> t) {
            initPropertyVaues();
            super.putAll(t);
        }

        @Override
        public synchronized void clear() {
            initPropertyVaues();
            super.clear();
        }

        @Override
        public synchronized Object clone() {
            initPropertyVaues();
            return super.clone();
        }

        @Override
        public synchronized String toString() {
            initPropertyVaues();
            return super.toString();
        }

        @Override
        public Set<Object> keySet() {
            initPropertyVaues();
            return super.keySet();
        }

        @Override
        public Set<Entry<Object, Object>> entrySet() {
            initPropertyVaues();
            return super.entrySet();
        }

        @Override
        public Collection<Object> values() {
            initPropertyVaues();
            return super.values();
        }

        @Override
        public synchronized boolean equals(final Object o) {
            initPropertyVaues();
            return super.equals(o);
        }

        @Override
        public synchronized int hashCode() {
            initPropertyVaues();
            return super.hashCode();
        }

    }

    private static ObjectMapper objectMapper;
    private static Charset UTF8_CHARSET = Charset.forName("UTF-8");

    protected final Resource resource;
    protected final ResourceID id;
    protected final Properties properties;

    static {
        objectMapper = new ObjectMapper();
        objectMapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, true).configure(
            SerializationFeature.WRAP_EXCEPTIONS, false);
    }

    /**
     * Instance of ClientUtilities for accessing helper methods for XSS support and generating URLs.
     */
    protected ClientUtilities clientUtils;

    /**
     * A list of properties that will be omitted when you getProperties for the resource. The default will omit
     * 'cq.*', 'ip', 'email', 'userAgent'. This list can contain regex.
     */
    protected final List<String> ignoredProperties = new ArrayList<String>(Arrays.asList("cq:.*", "ip", "email",
        "userAgent", "sling:.*"));
    protected List<Pattern> ignoredPropertiesPatterns;

    private final List<String> whiteList = Arrays.asList("cq:tags");

    /**
     * @param resource the resource for which this BaseSocialComponent will be created for.
     * @param clientUtils an instance of ClientUtilities for accessing helper methods for XSS support and generating
     *            URLs.
     */
    public BaseSocialComponent(final Resource resource, final ClientUtilities clientUtils) {
        this.resource = resource;
        this.id = new DefaultResourceID(resource);
        this.properties = new LazyProperties();
        this.clientUtils = clientUtils;
    }

    public static String buildJSONString(final Map<String, Object> data, final boolean tidy) throws JsonException {
        try {
            final JsonFactory f = new JsonFactory();
            f.setCharacterEscapes(new EscapeCloseScriptBlocks());
            final ByteArrayOutputStream bastream = new ByteArrayOutputStream();
            final JsonGenerator jgen = f.createGenerator(bastream, JsonEncoding.UTF8);
            if (tidy) {
                objectMapper.writerWithDefaultPrettyPrinter().writeValue(jgen, data);
            } else {
                objectMapper.writeValue(jgen, data);
            }
            return new String(bastream.toByteArray(), UTF8_CHARSET);
        } catch (final JsonProcessingException e) {
            throw new JsonException("Error converting map to JSON", e);
        } catch (final IOException e) {
            throw new JsonException("Error converting map to JSON", e);
        }
    }

    @Override
    public ResourceID getId() {
        return this.id;
    }

    @Override
    public String getResourceType() {
        return this.resource.getResourceType();
    }

    @Override
    public String getUrl() {
        return this.externalizeURL(this.resource.getPath());
    }

    @Override
    public Properties getProperties() {
        return this.properties;
    }

    private void initIgnoredPropertiesList() {
        ignoredPropertiesPatterns = new ArrayList<Pattern>(getIgnoredProperties().size());
        for (final String ignoredKey : this.getIgnoredProperties()) {
            ignoredPropertiesPatterns.add(Pattern.compile(ignoredKey));
        }

    }

    /**
     * This method determines whether a property should be omitted or not by checking against the ignore list.
     * @param key property name
     * @return true if this property is to be omitted, false otherwise.
     */
    protected boolean isKeyIgnored(final String key) {
        for (final Pattern ignoredKey : ignoredPropertiesPatterns) {
            if (ignoredKey.matcher(key).matches() && !whiteList.contains(key)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method returns the list of property names/regex that need to be omitted. Override this method to return a
     * more comprehensive list specific to your component.
     * @return a list of property names/regex that need to be omitted.
     */
    protected List<String> getIgnoredProperties() {
        return this.ignoredProperties;
    }

    @Override
    @JsonIgnore
    public String toJSONString(final boolean tidy) throws JsonException {
        try {
            final JsonFactory f = new JsonFactory();
            f.setCharacterEscapes(new EscapeCloseScriptBlocks());
            final ByteArrayOutputStream bastream = new ByteArrayOutputStream();
            final JsonGenerator jgen = f.createGenerator(bastream, JsonEncoding.UTF8);
            if (tidy) {
                objectMapper.writerWithDefaultPrettyPrinter().writeValue(jgen, this);
            } else {
                objectMapper.writeValue(jgen, this);
            }
            return new String(bastream.toByteArray(), UTF8_CHARSET);
        } catch (final JsonProcessingException e) {
            throw new JsonException("Error converting " + this.id + " to JSON", e);
        } catch (final IOException e) {
            throw new JsonException("Error converting " + this.id + " to JSON", e);
        }
    }

    @Override
    @JsonIgnore
    public Resource getResource() {
        return this.resource;
    }

    /**
     * @param path absolute path to the resource
     * @return an externalized URL to the resource
     */
    protected String externalizeURL(final String path) {
        if (this.clientUtils == null) {
            return path;
        }
        return this.clientUtils.externalLink(path);
    }

    @Override
    @SuppressWarnings("unchecked")
    @JsonIgnore
    public Map<String, Object> getAsMap() {
        return objectMapper.convertValue(this, Map.class);
    }

    @Override
    public SocialComponent getParentComponent() {
        return null;
    }

    @Override
    public SocialComponent getSourceComponent() {
        return this;
    }

    @Override
    public String getFriendlyUrl() {
        final SocialUtils socialUtils = clientUtils.getSocialUtils();
        if (socialUtils != null) {
            final Page page = socialUtils.getContainingPage(resource);
            if (page != null) {
                return clientUtils.externalLink(page.getPath(), false) + ".html";
            }

        }
        return null;
    }

    static private class EscapeCloseScriptBlocks extends CharacterEscapes {
        private final int[] escapes;

        public EscapeCloseScriptBlocks() {
            final int[] baseEscapes = standardAsciiEscapesForJSON();
            baseEscapes['<'] = CharacterEscapes.ESCAPE_STANDARD;
            baseEscapes['>'] = CharacterEscapes.ESCAPE_STANDARD;
            escapes = baseEscapes;
        }

        @Override
        public int[] getEscapeCodesForAscii() {
            return escapes;
        }

        @Override
        public SerializableString getEscapeSequence(final int arg0) {
            return null;
        }
    }
}
