/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.scf.core.resourcetree;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

/*
 * Resource type hierarchies don't really fit in a tree structure. The data structure managed by this class is
 * actually a list of trees (a grove). Each tree in the list manages a set of resource type nodes that have a common
 * ancestor. The root of the tree is always the most common (or the super-most) resource type.
 * Each node in a resource type tree has a list of child nodes that are a direct (this does not mean that the super
 * resource type of the child is always the parent, it only means that the parent is the closest super type
 * that is available) subtype. To find the closest match for a given resource type, if the specific resource type is not found,
 * it does a top down search (starting with the root) to find the closest parent.
 */
/**
 * A class that exposes a data structure that can be used to manage entries keyed by a resource type in a resource
 * type hierarchy. This makes it possible to search for values based on the hierarchy, and retrieve results based on
 * super type links when a particular resource type does not have an entry.
 * @author palanisw
 * @param <T> - the type of value being stored/associated with each resource type entry added to the tree.
 */
public class ResourceTypeTree<T> {
    private static class IncompatibleResourceTypeException extends Exception {
        private static final long serialVersionUID = 1L;

        public IncompatibleResourceTypeException(final String message) {
            super(message);
        }
    }

    private class RTree<T> {
        private Node<T> root;
        private boolean rootIsRemoved;
        private boolean rootChanged;

        public RTree(final String resourceType, final T value) {
            root = new Node<T>(resourceType, value, null);
        }

        public RTree(final Node<T> root) {
            this.root = root;
        }

        @Override
        public String toString() {
            return root == null ? null : root.toString();
        }

        @SuppressWarnings("unused")
        public Node<T> getRoot() {
            return root;
        }

        /**
         * @param resourceType the resource type to insert into this tree.
         * @param value the value to associate with the resource type being inserted.
         * @throws ResourceTypeTree.IncompatibleResourceTypeException if the resource type that is being inserted does
         *             not belong to the resource type hierarchy maintained by this tree.
         * @return boolean true if the root of this tree has been changed as a result of the insert, false otherwise
         */
        public boolean put(final String resourceType, final T value)
            throws ResourceTypeTree.IncompatibleResourceTypeException {
            if (root == null) {
                root = new Node<T>(resourceType, value, null);
                this.rootIsRemoved = false;
                return true;
            } else {
                insert(resourceType, value, root);
                if (this.rootChanged) {
                    this.rootChanged = false;
                    return true;
                }
                return false;
            }
        }

        private void insert(final String resourceType, final T value, final Node<T> node)
            throws ResourceTypeTree.IncompatibleResourceTypeException {
            final Resource nodeResource = ResourceTypeTree.this.adminResolver.getResource(node.getResourceType());
            if (ResourceTypeTree.this.adminResolver.isResourceType(nodeResource, resourceType)) {
                final Node<T> newNode = new Node<T>(resourceType, value, node.getParent());
                final Node<T> oldParent = node.getParent();
                node.setParent(newNode);
                newNode.addChild(node);
                if (node == root) {
                    root = newNode;
                    this.rootChanged = true;
                }
                if (oldParent != null) {
                    oldParent.removeChild(node);
                    final List<Node<T>> siblings = new ArrayList<Node<T>>(oldParent.getChildren());
                    for (final Node<T> sibling : siblings) {
                        final Resource siblingResource =
                            ResourceTypeTree.this.adminResolver.getResource(sibling.getResourceType());
                        if (ResourceTypeTree.this.adminResolver.isResourceType(siblingResource, resourceType)) {
                            sibling.setParent(newNode);
                            newNode.addChild(sibling);
                            oldParent.removeChild(sibling);
                        }
                    }
                    oldParent.addChild(newNode);
                }

            } else {
                final Resource resourceToAdd = ResourceTypeTree.this.adminResolver.getResource(resourceType);
                if (!ResourceTypeTree.this.adminResolver.isResourceType(resourceToAdd, node.getResourceType())) {
                    throw new IncompatibleResourceTypeException(resourceToAdd + " is not a " + node.getResourceType());
                }
                boolean isGrandchild = false;
                for (final Node<T> child : node.getChildren()) {
                    try {
                        insert(resourceType, value, child);
                        isGrandchild = true;
                        break;
                    } catch (final IncompatibleResourceTypeException e) {
                        continue;
                    }
                }
                if (!isGrandchild) {
                    final Node<T> child = new Node<T>(resourceType, value, node);
                    node.addChild(child);
                }
            }
        }

        public boolean isEmpty() {
            return root == null;
        }

        /**
         * @param resourceType the resource type to be removed from the tree.
         * @return if the resource type that was removed was the root if this tree, then this method returns a list of
         *         children of the root so that they can be added to the grove as individual trees. Returns an empty
         *         list if the resource type removed was not the root of the tree.
         */
        public List<RTree<T>> remove(final String resourceType) {
            delete(resourceType, root);
            if (this.rootIsRemoved) {
                final List<RTree<T>> newTrees = new ArrayList<RTree<T>>(root.getChildren().size());
                for (final Node<T> newSubRoot : root.getChildren()) {
                    newSubRoot.setParent(null);
                    newTrees.add(new RTree<T>(newSubRoot));
                }
                root = null;
                return newTrees;
            }
            return Collections.<RTree<T>>emptyList();
        }

        private boolean delete(final String resourceType, final Node<T> node) {
            final Resource resource = ResourceTypeTree.this.adminResolver.getResource(resourceType);
            if (StringUtils.equals(resourceType, node.getResourceType())) {
                final Node<T> parent = node.getParent();
                if (parent == null) {
                    this.rootIsRemoved = true;
                } else {
                    for (final Node<T> child : node.getChildren()) {
                        child.setParent(parent);
                        parent.addChild(child);
                    }
                    parent.removeChild(node);
                }
                return true;
            } else {
                if (!ResourceTypeTree.this.adminResolver.isResourceType(resource, node.getResourceType())) {
                    return false;
                }
                boolean wasChild = false;
                for (final Node<T> child : node.getChildren()) {
                    wasChild = delete(resourceType, child);
                    if (wasChild) {
                        return true;
                    }
                }
            }
            return false;
        }

        public Entry<String, T> findClosestParent(final Resource resourceToFind) {

            if (!ResourceTypeTree.this.adminResolver.isResourceType(resourceToFind, root.getResourceType())) {
                return null;
            }
            final Node<T> closest = findClosestParentNode(root, resourceToFind);
            final Entry<String, T> closestEntry = new Entry<String, T>() {

                @Override
                public String getKey() {
                    return closest.getResourceType();
                }

                @Override
                public T getValue() {
                    return closest.getValue();
                }

                @Override
                public T setValue(final T arg0) {
                    return null;
                }
            };
            return closestEntry;
        }

        public Collection<Entry<String, T>> getParents(final Resource resource) {
            if (!ResourceTypeTree.this.adminResolver.isResourceType(resource, root.getResourceType())) {
                return null;
            }
            Collection<Entry<String, T>> parents = new ArrayList<Entry<String, T>>();
            parents = getParents(resource, root, parents);
            return parents;
        }

        protected Collection<Entry<String, T>> getParents(final Resource resource, final Node<T> root,
            final Collection<Entry<String, T>> parents) {

            final Entry<String, T> rootEntry = new RTreeEntry(root);
            parents.add(rootEntry);
            Node<T> parent = root;
            for (final Node<T> child : parent.getChildren()) {
                if (child.getResourceType().equals(resource.getResourceType()))
                    break;
                if (ResourceTypeTree.this.adminResolver.isResourceType(resource, child.getResourceType())) {
                    getParents(resource, child, parents);
                }
            }
            return parents;
        }

        private class RTreeEntry implements Entry<String, T> {
            final Node<T> node;

            public RTreeEntry(Node<T> node) {
                this.node = node;
            }

            @Override
            public String getKey() {
                return node.getResourceType();
            }

            @Override
            public T getValue() {
                return node.getValue();
            }

            @Override
            public T setValue(T value) {
                return null;
            }

        }

        private Node<T> findClosestParentNode(final Node<T> node, final Resource resource) {
            Node<T> closest = node;
            boolean bottomedOut = false;
            while (closest.hasChildren() && !bottomedOut) {
                bottomedOut = true;
                for (final Node<T> child : closest.getChildren()) {
                    if (ResourceTypeTree.this.adminResolver.isResourceType(resource, child.getResourceType())) {
                        bottomedOut = false;
                        closest = child;
                        break;
                    }
                }
            }
            return closest;
        }

        private class Node<T> {
            private final String resourceType;
            private final T value;
            private Node<T> parent;
            private List<Node<T>> children;

            public Node(final String resourceType, final T value, final Node<T> parent) {
                this.resourceType = resourceType;
                this.value = value;
                this.parent = parent;
                children = new ArrayList<Node<T>>(5);
            }

            @Override
            public String toString() {
                return resourceType;
            }

            public boolean hasChildren() {
                return children.size() != 0;
            }

            public void removeChild(final Node<T> node) {
                children.remove(node);
            }

            public void setParent(final Node<T> parent) {
                this.parent = parent;
            }

            public String getResourceType() {
                return resourceType;
            }

            public List<Node<T>> getChildren() {
                return children;
            }

            public Node<T> getParent() {
                return parent;
            }

            public T getValue() {
                return value;
            }

            @SuppressWarnings("unused")
            public void setChildren(final List<Node<T>> children) {
                this.children = children;
            }

            public void addChild(final Node<T> child) {
                this.children.add(child);
            }
        }

    }

    private ResourceResolver adminResolver;

    private final Map<String, T> hash;
    private final List<RTree<T>> resourceTypeTrees;

    /**
     * @param adminResolver an instance of the {@link ResourceResolver} with administrative privileges that will be
     *            used to compare resource types.
     */
    public ResourceTypeTree(final ResourceResolver adminResolver) {
        this.adminResolver = adminResolver;
        hash = new HashMap<String, T>(10);
        resourceTypeTrees = new ArrayList<RTree<T>>(10);
    }

    /**
     * @param adminResolver an instance of the {@link ResourceResolver} with administrative privileges that will be
     *            used to compare resource types.
     */
    public synchronized void setAdminResolver(final ResourceResolver adminResolver) {
        this.adminResolver = adminResolver;
    }

    /**
     * @param resourceType the resource type to search for.
     * @return true if the specific resource type is present, false otherwise. Does not search up or down the tree.
     */
    public boolean contains(final String resourceType) {
        return hash.containsKey(resourceType);
    }

    /**
     * Adds the given resource type and value to the resource type trees. Replaces the entry if an entry already
     * exists for the given resource type.
     * @param resourceType resource type to add.
     * @param value value to associate with the resource type.
     */
    public synchronized void put(final String resourceType, final T value) {
        hash.put(resourceType, value);
        boolean added = false;
        boolean recheckGrove = false;
        RTree<T> treeAddedTo = null;
        for (final RTree<T> tree : resourceTypeTrees) {
            try {
                recheckGrove = tree.put(resourceType, value);
                treeAddedTo = tree;
                added = true;
                break;
            } catch (final IncompatibleResourceTypeException e) {
                continue;
            }
        }
        if (!added) {
            final RTree<T> newTree = new RTree<T>(resourceType, value);
            resourceTypeTrees.add(newTree);
        }
        if (recheckGrove) {
            final Iterator<RTree<T>> iterator = resourceTypeTrees.iterator();
            while (iterator.hasNext()) {
                final RTree<T> tree = iterator.next();
                if (tree != treeAddedTo) {
                    final Resource treeRoot = this.adminResolver.getResource(tree.getRoot().getResourceType());
                    if (this.adminResolver.isResourceType(treeRoot, treeAddedTo.getRoot().getResourceType())) {
                        treeAddedTo.getRoot().addChild(tree.getRoot());
                        tree.getRoot().setParent(treeAddedTo.getRoot());
                        iterator.remove();
                    }
                }
            }
        }
    }

    /**
     * Removes the given resource type (if present) from the resource type trees built up.
     * @param resourceType resource type to remove.
     */
    public synchronized void remove(final String resourceType) {
        if (hash.containsKey(resourceType)) {
            final Iterator<RTree<T>> iterator = resourceTypeTrees.iterator();
            final List<RTree<T>> newTrees = new ArrayList<RTree<T>>();
            while (iterator.hasNext()) {
                final RTree<T> tree = iterator.next();
                newTrees.addAll(tree.remove(resourceType));
                if (tree.isEmpty()) {
                    iterator.remove();
                }
            }
            resourceTypeTrees.addAll(newTrees);
        }
        hash.remove(resourceType);
    }

    /**
     * Gets the most closest matched value for a given resource type. If the specific resource type is not found,
     * searches up the resource type hierarchy to find the closest parent for the resource type.
     * @param resourceType resource type to search for
     * @return the value of the resource type if added. The value of the closest parent if the resource type is not
     *         found. Returns null if no parent of the resource type is found.
     */
    public T getClosest(final String resourceType) {
        if (hash.containsKey(resourceType)) {
            return hash.get(resourceType);
        }
        Entry<String, T> closestParent = null;
        final Resource resourceToFind = getResourceToFind(resourceType);
        if (resourceToFind == null) {
            return null;
        }
        for (final RTree<T> tree : resourceTypeTrees) {
            closestParent = tree.findClosestParent(resourceToFind);
            if (closestParent != null) {
                break;
            }
        }
        return closestParent == null ? null : closestParent.getValue();
    }

    private Resource getResourceToFind(final String resourceType) {
        final Resource resource = this.adminResolver.getResource(resourceType);
        if (resource == null) {
            return resource;
        }
        if (resource.getResourceSuperType() == null && StringUtils.startsWith(resource.getPath(), "/apps/")) {
            final String libsPath = "/libs/" + StringUtils.removeStart(resource.getPath(), "/apps/");
            final Resource libsResource = this.adminResolver.getResource(libsPath);
            if (libsResource != null) {
                return libsResource;
            }
        }
        return resource;
    }

    /**
     * Gets the specific value added for a given specific resource type. Does not search the heirarchy chain.
     * @param resourceType the resource type to search for.
     * @return the value added for the given resource type if found, null if the resource type was not found.
     */
    public T get(final String resourceType) {
        return hash.get(resourceType);
    }

    /**
     * Retrieves a list of all the children entries for a resource type based on the resource type hierarchy. Not yet
     * implemented.
     * @param resourceType the resource type for which parents will be found.
     * @return a list of entries listing each child in the hierarchy for the given resource type.
     */
    public List<Entry<String, T>> getChildren(final String resourceType) {
        throw new UnsupportedOperationException("Not yet implemented.");
    }

    /**
     * Retrieves a list of all the parent entries for a resource type based on the resource type hierarchy.
     * @param resourceType the resource type for which parents will be found.
     * @return a list of entries listing each parent in the hierarchy for the given resource type.
     */
    public Collection<Entry<String, T>> getParents(final String resourceType) {
        Entry<String, T> closestParent = null;
        final Resource resourceToFind = getResourceToFind(resourceType);
        if (resourceToFind == null) {
            return null;
        }
        for (final RTree<T> tree : resourceTypeTrees) {
            Collection<Entry<String, T>> parents = tree.getParents(resourceToFind);
            if (parents != null) {
                return parents;
            }
        }
        return null;
    }

    /**
     * Clears the resource type hierarchy that has been built.
     */
    public void clear() {
        hash.clear();
        this.resourceTypeTrees.clear();
    }

}
