/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.srp;

import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.sling.api.resource.ModifyingResourceProvider;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.QueriableResourceProvider;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceProvider;
import org.apache.sling.api.resource.ResourceResolver;

import com.adobe.cq.social.srp.config.SocialResourceConfiguration;

/** */
public interface SocialResourceProvider extends ResourceProvider, ModifyingResourceProvider,
    QueriableResourceProvider {

    /** Used in factory. */
    String SOLRQUERY = "SOLRQUERY";

    /** JSRP Query language. */
    String JSRPQUERY = "JSRP_SQL";

    /** User preferred language param, read from user profile */
    String PARAM_USER_PREFFERED_LANGUAGE = "ul";

    /** browser language param. */
    String PARAM_BROWSER_LANGUAGE = "bl";

    /** page language param. */
    String PARAM_PAGE_LANGUAGE = "pl";

    /** expected language param, this is explicitly set in the search widget, acts as a user override. */
    String PARAM_EXPECTED_LANGUAGE = "el";

    /**
     * Get a list of the children of the resource at the path.
     * @param path path to parent
     * @param resourceResolver resolver to use
     * @param offset offset to start at
     * @param size number to return
     * @param sortBy order to return children in
     * @return iterator of the child resources
     */
    Iterator<Resource> listChildren(String path, ResourceResolver resourceResolver, int offset, int size,
        List<Entry<String, Boolean>> sortBy);

    /**
     * Get a list of the children of the resource at the path. Filtered by child base type.
     * @param path path to parent
     * @param baseType base type of children to return (SocialUtils.PN_BASETYPE value) can be null/empty
     * @param resourceResolver resolver to use
     * @param offset offset to start at
     * @param size number to return
     * @param sortBy order to return children in
     * @return iterator of the child resources
     */
    Iterator<Resource> listChildren(String path, String baseType, ResourceResolver resourceResolver, int offset,
        int size, List<Entry<String, Boolean>> sortBy);

    /**
     * Get a list of the children of the resource at the path. Filtered by child base type.
     * @param path path to parent
     * @param baseType base type of children to return (SocialUtils.PN_BASETYPE value) can be null/empty
     * @param resourceResolver resolver to use
     * @param offset offset to start at
     * @param size number to return
     * @param sortBy order to return children in
     * @param visibleOnly whether to (true) return only visible documents, or (false) all documents
     * @return iterator of the child resources
     */
    Iterator<Resource> listChildren(String path, String baseType, ResourceResolver resourceResolver, int offset,
        int size, List<Entry<String, Boolean>> sortBy, boolean visibleOnly);

    /**
     * return count of all children (one level).
     * @param parent the parent resource
     * @return The all children count or the visible children count
     */
    long countChildren(Resource parent);

    /**
     * return count of all children of specific type (one level). Filtered by child base type.
     * @param parent the parent resource
     * @param baseType The social base type (SocialUtils.PN_BASETYPE value) of children to count (can be null/empty)
     * @return The all children count or the visible children count
     */
    long countChildren(Resource parent, String baseType);

    /**
     * return count of all children or filtered (visible) children (not hidden due to denied/flagged as hidden). One
     * level of children only
     * @param parent the parent resource
     * @param visibleOnly if true, count only the 'visible' children
     * @return The all children count or the visible children count
     */
    long countChildren(Resource parent, boolean visibleOnly);

    /**
     * return count of all children or filtered (visible) children (not hidden due to denied/flagged as hidden).
     * filter by the child base type. One level of children only
     * @param parent the parent resource
     * @param baseType The social base type (SocialUtils.PN_BASETYPE value) of children to count (can be null/empty)
     * @param visibleOnly if true, count only the 'visible' children
     * @return The all children count or the visible children count
     */
    long countChildren(Resource parent, String baseType, boolean visibleOnly);

    /**
     * return counts of all children or filtered (visible) children (not hidden due to denied/flagged as hidden).
     * filter by the child base type. One level of children only
     * @param parent the parent resources
     * @param baseType The social base type (SocialUtils.PN_BASETYPE value) of children to count (can be null/empty)
     * @param visibleOnly if true, count only the 'visible' children
     * @return The all children counts or the visible children counts
     */
    Map<String, Long> countChildren(List<Resource> parent, String baseType, boolean visibleOnly);

    /**
     * Set the configuration for the provider.
     * @param config The configuration
     */
    void setConfig(SocialResourceConfiguration config);

    /**
     * Get the ASI path (provider root).
     * @return The provider root.
     */
    String getASIPath();

    /**
     * Return a list of related posts, given a query.
     * @param resolver The resource resolver.
     * @param query The string to search for.
     * @param statusFilter The filter on the status.
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component we want to search on.
     * @param mltFields Resource properties to search on. NOTE: Only first one is used
     * @param maxResults The maximum results returned.
     * @param minTermFreq The frequency which terms will be ignored in the source doc
     * @param minDocFreq The frequency at which words will be ignored which do not occur in at least this many docs.
     * @return The list of matching documents. Or an empty list if no documents are found or the Solr server cannot be
     *         queried.
     * @deprecated Solr and Lucene only use the first mltField when using a stream as the document (our use case) See:
     *             https://issues.apache.org/jira/browse/SOLR-5351?
     */
    @Deprecated
    Iterator<Resource> getMLTResults(ResourceResolver resolver, String query, String statusFilter,
        String resourceTypeFilter, String componentFilter, String[] mltFields, int maxResults, int minTermFreq,
        int minDocFreq);

    /**
     * Return a list of related posts, given a query.
     * @param resolver The resource resolver.
     * @param query The string to search for.
     * @param statusFilter The filter on the status.
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component we want to search on.
     * @param mltField Resource property to search on.
     * @param maxResults The maximum results returned.
     * @param minTermFreq The frequency which terms will be ignored in the source doc
     * @param minDocFreq The frequency at which words will be ignored which do not occur in at least this many docs.
     * @return The list of matching documents. Or an empty list if no documents are found or the Solr server c cannot
     *         be queried.
     */
    Iterator<Resource> getMLTResults(ResourceResolver resolver, String query, String statusFilter,
        String resourceTypeFilter, String componentFilter, String mltField, int maxResults, int minTermFreq,
        int minDocFreq);

    /**
     * Return a result list of resources given a component and a query.
     * @param resolver The resource resolver.
     * @param component The component for filtering on
     * @param query The lucene query
     * @param sortFields fields to use in sort and sort direction
     * @param offset starting offset to return results at
     * @param limit maximum number of results to return
     * @param requiresTotal true iff the total number of matches needs to be returned in the
     *            SocialResourceSearchResult
     * @return The matching documents.
     */
    SocialResourceSearchResult<Resource> find(ResourceResolver resolver, String component, String query,
        List<SearchSortField> sortFields, int offset, int limit, boolean requiresTotal);

    /**
     * Return a result list of resources given a component and a query.
     * @param resolver The resource resolver.
     * @param component The component for filtering on
     * @param query The lucene query
     * @param sortFields fields to use in sort and sort direction
     * @param offset starting offset to return results at
     * @param limit maximum number of results to return
     * @param requiresTotal true iff the total number of matches needs to be returned in the
     *            SocialResourceSearchResult
     * @param fieldLanguage search will be queried against verbatim_fieldLanguage. This can be changed in the SOLR
     *            request handler.
     * @param signals Map containing field names with a map of values, user preferred language, browser language, page
     *            language and user selected language in the search widget. These values are used by SOLR request
     *            handler to determine the language of query string.
     * @return The matching documents.
     */
    SocialResourceSearchResult<Resource> find(ResourceResolver resolver, String component, String query,
        List<SearchSortField> sortFields, int offset, int limit, boolean requiresTotal, String fieldLanguage,
        Map<String, Object> signals);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param fieldNames Fields for which facet counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @param visibleOnly whether to filter by visibility
     * @return Map containing field names with a map of values and counts for each field
     */
    Map<String, Map<String, Integer>> findFacets(ResourceResolver resolver, List<String> fieldNames,
        String resourceTypeFilter, String componentFilter, int count, final boolean visibleOnly);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param fieldNames Fields for which facet counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @return Map containing field names with a map of values and counts for each field
     */
    Map<String, Map<String, Integer>> findFacets(ResourceResolver resolver, List<String> fieldNames,
        String resourceTypeFilter, String componentFilter, int count);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param fieldNames Fields for which facet counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @param visibleOnly whether to filter by visibility
     * @param includeChildren Include subpaths for search
     * @return Map containing field names with a map of values and counts for each field
     */
    Map<String, Map<String, Integer>> findFacets(ResourceResolver resolver, List<String> fieldNames,
        String resourceTypeFilter, String componentFilter, int count, boolean isVisibleOnly, boolean includeChildren);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param fieldNames Fields for which facet counts wanted
     * @param facetRanges Fields for which facet range counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @param visibleOnly whether to filter by visibility
     * @return Map containing field names with a map of values and counts for each field
     */
    FacetSearchResult findFacets(ResourceResolver resolver, List<String> fieldNames,
        List<FacetRangeField> facetRanges, String resourceTypeFilter, String componentFilter, int count,
        final boolean visibleOnly);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param fieldNames Fields for which facet counts wanted
     * @param facetRanges Fields for which facet range counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @param visibleOnly whether to filter by visibility
     * @param includeChildren Include results from paths subtree
     * @return Map containing field names with a map of values and counts for each field
     */
    FacetSearchResult findFacets(ResourceResolver resolver, List<String> fieldNames,
        List<FacetRangeField> facetRanges, String resourceTypeFilter, String componentFilter, int count,
        final boolean visibleOnly, final boolean includeChildren);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param fieldNames Fields for which facet counts wanted
     * @param facetRanges Fields for which facet range counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @return Map containing field names with a map of values and counts for each field
     */
    FacetSearchResult findFacets(ResourceResolver resolver, List<String> fieldNames,
        List<FacetRangeField> facetRanges, String resourceTypeFilter, String componentFilter, int count);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param facetRanges Fields for which facet range counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @param visibleOnly whether to filter by visibility
     * @return Map containing field names with a map of values and counts for each field
     */
    Map<String, Map<String, Integer>> findFacetRanges(ResourceResolver resolver, List<FacetRangeField> facetRanges,
        String resourceTypeFilter, String componentFilter, int count, final boolean visibleOnly);

    /**
     * Search Collections findFacets for cloud storage resources.
     * @param resolver ResourceResolver for current search permissions
     * @param facetRanges Fields for which facet range counts wanted
     * @param resourceTypeFilter Resource type filter
     * @param componentFilter Component filter
     * @param count maximum results to return
     * @return Map containing field names with a map of values and counts for each field
     */
    Map<String, Map<String, Integer>> findFacetRanges(ResourceResolver resolver, List<FacetRangeField> facetRanges,
        String resourceTypeFilter, String componentFilter, int count);

    /**
     * Return a list of the languages that this provider can use to query resources.
     * @return The list of valid languages.
     */
    List<String> getLanguages();

    /**
     * Return the content type for pulling.
     * @return the content type as a string.
     */
    String getContentType();

    /**
     * Add one to the specified counter property.
     * @param resource - the resource to update.
     * @param property - the name of the counter.
     * @throws PersistenceException on error
     */
    void increment(Resource resource, String property) throws PersistenceException;

    /**
     * Subtract one from the specified counter property.
     * @param resource - the resource to update.
     * @param property - the name of the counter.
     * @throws PersistenceException on error
     */
    void decrement(Resource resource, String property) throws PersistenceException;

    /**
     * Add an arbitrary increment to the specified counter property.
     * @param resource - the resource to update.
     * @param property - the name of the counter.
     * @param increment - the amount to increase or decrease the counter.
     * @throws PersistenceException on error
     */
    void incrementBy(Resource resource, String property, Long increment) throws PersistenceException;

    /**
     * Add an arbitrary increment to the specified properties.
     * @param resource the resource.
     * @param incrementMap map with properties and their increment/decrement values.
     * @throws PersistenceException on error
     */
    void incrementBy(Resource resource, Map<String, Long> incrementMap) throws PersistenceException;

    /**
     * Return the value of the counter.
     * @param resource - the resource to query.
     * @param property - the name of the counter.
     * @return the count
     */
    long getCount(Resource resource, String property);

    /**
     * Gets details about the provider.
     * @see ProviderMetaData
     * @return the provider's metadata
     */
    ProviderMetaData getMetaData();

    /**
     * Fetch a group of resources all managed by this resource provider
     * @param resolver the resolver to use
     * @param resources the list of resources
     * @return a map from path to Resource (which will be a NonExistingResource if the resource could not be fetched).
     */
    Map<String, Resource> getResources(ResourceResolver resolver, List<String> resources);

    /**
     * Return an attachment resource as an InputStream.
     * @param resolver resolver to use to read the resource
     * @param path path to the attachment resource
     * @return the attachment as an InputStream
     * @throws IOException on error
     */
    InputStream getAttachmentInputStream(ResourceResolver resolver, String path) throws IOException;

    Resource move(final ResourceResolver resolver, final String srcAbsPath, final String destAbsPath)
        throws PersistenceException;
}
