/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.srp.internal;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ModifiableValueMapDecorator;

/**
 * <code>AdobeSocialModifiableValueMap</code> decorates another {@link Map} to provide a basic implementation for the
 * additional methods of a {@link ModifiableValueMapDecorator}.
 * @since 0.0.1
 */
class SocialModifiableValueMap extends ModifiableValueMapDecorator {
    private static final Set<String> KEY_BLACK_LIST;

    static {
        KEY_BLACK_LIST = new HashSet<String>();
        KEY_BLACK_LIST.add(AbstractSchemaMapper.getSocoKey());
        KEY_BLACK_LIST.add(AbstractSchemaMapper.getSocoParentIdKey());
    }

    private final MapResourceImpl resource;

    /**
     * Creates a new wrapper around a given map.
     * @param resource wrapped object
     */
    public SocialModifiableValueMap(final MapResourceImpl resource) {
        super(resource.adaptTo(Map.class));
        this.resource = resource;
    }

    @Override
    public Object put(final String key, final Object value) {
        if (KEY_BLACK_LIST.contains(key)) {
            return null;
        }
        resource.update();
        if (super.containsKey(CachingResourceProvider.INC) && super.get(CachingResourceProvider.INC) instanceof Map) {
            final Map<String, Long> incMap = (Map<String, Long>) super.get(CachingResourceProvider.INC);
            if (incMap.containsKey(key)) {
                incMap.remove(key); // "set" overrides "increment", if applied after
                if (incMap.isEmpty()) {
                    super.remove(CachingResourceProvider.INC);
                }
            }
        }
        return super.put(key, value);
    }

    @Override
    public Object remove(final Object key) {
        if (KEY_BLACK_LIST.contains(key) || !super.containsKey(key) || super.get(key) == null) {
            return null;
        }

        resource.update();
        // AS can't remove properties, so set to null to indicate that the property is removed. We'll remove null
        // properties from resources before sending back up to caller
        return super.put((String) key, null);
    }

    @Override
    public void clear() {
        super.clear();
        resource.update();
    }

    @Override
    public void putAll(final Map<? extends String, ? extends Object> m) {
        boolean dirty = false;
        for (final String key : m.keySet()) {
            if (KEY_BLACK_LIST.contains(key)) {
                continue;
            }
            super.put(key, m.get(key));
            dirty = true;
        }
        if (dirty) {
            resource.update();
        }
    }

    @Override
    public <T> T get(String name, Class<T> type) {
        final T value = super.get(name, type);
        if ((type == Integer.class || type == Long.class)
                && (super.containsKey(CachingResourceProvider.INC) && super.get(CachingResourceProvider.INC) instanceof Map)) {
            Map incMap = (Map) super.get(CachingResourceProvider.INC);
            if (incMap.containsKey(name) && incMap.get(name) instanceof Long) {
                if (type == Long.class) {
                    Long longValue = (Long) value;
                    longValue += (Long) incMap.get(name);
                    return (T) longValue;
                } else {
                    Integer intValue = (Integer) value;
                    intValue += ((Long) incMap.get(name)).intValue();
                    return (T) intValue;
                }
            }
        }
        return value;
    }

}
