/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.srp.utilities.api;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

import com.adobe.cq.social.srp.SocialResourceProvider;
import com.adobe.cq.social.srp.config.SocialResourceConfiguration;

public interface SocialResourceUtilities {

    /**
     * This method maps the given <code>resource</code> to its equivalent path in UGC for ACL checking. For cloud
     * storage a component like: <i>/content/site/en/page/jcr:content/par/forum</i> will map to
     * <i>/content/usergenerated/content/site/en/page/</i>. For JCR content,
     * <i>/content/site/en/page/jcr:content/par/forum</i> will map to:
     * <i>/content/usergenerated/content/site/en/page/</i> Note: This method maps the Cloud Storage path to the JCR
     * marker used for ACL checking, not the path that is used as the resource path for the user generated content
     * itself
     * @param resource The resource to transpose to the comment's UCG path.
     * @return A <code>String</code> representing the equivalent UCG path.
     */
    @CheckForNull
    String resourceToACLPath(@Nullable final Resource resource);

    /**
     * This method maps the given <code>resource</code> to its equivalent UGC page path. For cloud storage a component
     * like: <i>/content/site/en/page/jcr:content/par/forum</i> will map to
     * <i>/content/usergenerated/asi/cloud/content/site/en/page/</i>. For JCR content,
     * <i>/content/site/en/page/jcr:content/par/forum</i> will map to:
     * <i>/content/usergenerated/content/site/en/page/</i> Note: A cloud path can not be used for ACL checking. See
     * resourceToACLPath for mapping the resource to a UGC path suitable for ACL checking (returns the 'marker' in
     * /content/usergenerated/content)
     * @param resource The resource to transpose to the cloud UGC path.
     * @return A <code>String</code> representing the equivalent cloud UCG path.
     */
    @CheckForNull
    String resourceToUGCStoragePath(@Nullable final Resource resource);

    /**
     * This method maps the given <code>resource</code> in /content/usergenerated to its equivalent path in the normal
     * content tree, e.g. as the relationship of a comment in <i>//content/usergenerated/content/test</i> to its
     * parent resource at <i>/content/test</i>.
     * @param resource The resource to transpose.
     * @return A <code>String</code> representing the equivalent normal content path.
     */
    String ugcToResourcePath(@Nonnull final Resource resource);

    /**
     * This method maps the given <code>resource</code> in /content/usergenerated to its equivalent path in the normal
     * content tree, e.g. as the relationship of a comment in <i>//content/usergenerated/content/test</i> to its
     * parent resource at <i>/content/test</i>.
     * @param ugcPath The path to transpose.
     * @return A <code>String</code> representing the equivalent normal content path.
     */
    @CheckForNull
    String ugcToResourcePath(@Nullable final String ugcPath);

    /**
     * Gets the base UGC Resource from a SocialResource. This is used to get the JCR base node of a remote piece of
     * UGC. It will also create all the parallel content/usergenerated paths.
     * @param userResource a path to a resource that needs to have its UGC Resource located.
     * @return the path to the UGC location for this social resource
     */
    @CheckForNull
    Resource getUGCResource(final Resource userResource);

    /**
     * Gets the base UGC Resource from a SocialResource. This is used to get the JCR base node of a remote piece of
     * UGC. It will also create all the parallel content/usergenerated paths.
     * @param userResource a path to a resource that needs to have its UGC Resource located.
     * @param resourceTypeHint the resource type to set on the root node in the UGC tree, this is used when components
     *            are cq:included onto a page
     * @return the path to the UGC location for this social resource
     */
    @CheckForNull
    Resource getUGCResource(final Resource userResource, final String resourceTypeHint);

    /**
     * From a resource get the configuration for the social data service.
     * @param resource the base resource to get the configuration for
     * @return a configuration for the social data store or null if one does not exist
     */
    @CheckForNull
    SocialResourceConfiguration getStorageConfig(final Resource resource);

    /**
     * Get a SocialResourceProvider given a resource.
     * @param resource The resource to return the provider for.
     * @return social resource provider, if a provider wasn't found it will return the default
     */
    @Nonnull
    SocialResourceProvider getSocialResourceProvider(@Nullable final Resource resource);

    /**
     * Check if the login user can moderate a given resource. This will always return false if the resource is not
     * under the UGC tree.
     * @param resource The resource to check
     * @return true iff the login user can moderate a given resource.
     */
    boolean hasModeratePermissions(@Nonnull final Resource resource);

    /**
     * Checks a path and an action based on the passed in resolver.
     * @param resolver the resource resolver to use to evaluate the permissions
     * @param path the path to the resource to check
     * @param action the action to check
     * @return True if the resolver is allowed to take the requested action, otherwise false
     */
    Boolean checkPermission(@Nonnull final ResourceResolver resolver, final String path, final String action);
}
