/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.subscriptions.endpoint;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Session;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.request.RequestParameterMap;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.SyntheticResource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.graph.SocialGraph;
import com.adobe.cq.social.graph.Vertex;
import com.adobe.cq.social.scf.ClientUtilities;
import com.adobe.cq.social.scf.Operation;
import com.adobe.cq.social.scf.OperationException;
import com.adobe.cq.social.scf.OperationExtension;
import com.adobe.cq.social.scf.SocialComponent;
import com.adobe.cq.social.scf.SocialComponentFactory;
import com.adobe.cq.social.scf.SocialComponentFactoryManager;
import com.adobe.cq.social.scf.core.operations.AbstractOperationService;
import com.adobe.cq.social.subscriptions.api.SubscriptionManager;
import com.adobe.cq.social.subscriptions.api.SubscriptionType;
import com.adobe.cq.social.subscriptions.client.api.SubscriptionCollection;
import com.adobe.cq.social.subscriptions.client.api.SubscriptionCollectionComponentFactory;
import com.adobe.cq.social.subscriptions.endpoint.SubscriptionCollectionOperationExtension.SubscriptionCollectionOperation;
import com.adobe.cq.social.ugcbase.CollabUser;
import com.adobe.granite.socialgraph.Direction;
import com.adobe.granite.socialgraph.Relationship;

/**
 * Provides abstract implementation of the Subscription operation..
 * @param <T> is a {@link OperationExtension} that will be used as hooks by the extending class.
 * @param <U> is a {@link Operation} that is being provided by the extending class.
 */
@Component(metatype = false, componentAbstract = true)
public abstract class AbstractSubscriptionCollectionOperationService<T extends SubscriptionCollectionOperationExtension, U extends SubscriptionCollectionOperation>
    extends AbstractOperationService<T, U, SubscriptionCollection> implements SubscriptionCollectionOperations {
    @Reference
    SubscriptionManager subscriptionManager;
    @Reference
    private SocialComponentFactoryManager componentFactoryManager;

    String subscribedId;
    String userId;
    String types[];
    Boolean states[];
    ResourceResolver resolver;
    private static List<String> PARAM_LIST = Arrays.asList(new String[]{PROP_SUBSCRIBED_ID, PROP_USER_ID, PROP_TYPES,
        PROP_STATES, ":operation"});
    private static Logger LOG = LoggerFactory.getLogger(AbstractSubscriptionCollectionOperationService.class);

    @Override
    public SocialComponent update(final SlingHttpServletRequest request) throws OperationException {
        resolver = request.getResource().getResourceResolver();
        final Map<String, Object> requestParams = new HashMap<String, Object>();
        // Validate parameters
        userId = resolver.getUserID();
        final boolean userIsLoggedIn =
            !(StringUtils.isEmpty(userId) || userId.equalsIgnoreCase(CollabUser.ANONYMOUS));
        if (!userIsLoggedIn) {
            throw new OperationException("User needs to login", HttpServletResponse.SC_FORBIDDEN);
        }
        subscribedId = request.getParameter(PROP_SUBSCRIBED_ID);
        if (StringUtils.isEmpty(subscribedId)) {
            throw new OperationException("Invalid '" + PROP_SUBSCRIBED_ID + "' value",
                HttpServletResponse.SC_BAD_REQUEST);
        }
        types = request.getParameterValues(PROP_TYPES);
        if (types == null || types.length == 0) {
            throw new OperationException("Invalid '" + PROP_TYPES + "' value", HttpServletResponse.SC_BAD_REQUEST);
        }
        // Verify types
        for (String type : types) {
            SubscriptionType subType = SubscriptionType.getByRelationshipId(type);
            if (subType == null) {
                throw new OperationException("Invalid '" + PROP_TYPES + ": " + type,
                    HttpServletResponse.SC_BAD_REQUEST);
            }
        }
        String reqStates[] = request.getParameterValues(PROP_STATES);
        if (reqStates == null || reqStates.length != types.length) {
            throw new OperationException("Invalid '" + PROP_STATES + "' value", HttpServletResponse.SC_BAD_REQUEST);
        }
        states = new Boolean[reqStates.length];
        for (int i = 0; i < reqStates.length; i++) {
            states[i] = Boolean.valueOf(reqStates[i]);
        }
        requestParams.put(PROP_SUBSCRIBED_ID, subscribedId);
        requestParams.put(PROP_USER_ID, userId);
        requestParams.put(PROP_TYPES, types);
        requestParams.put(PROP_STATES, states);
        final RequestParameterMap params = request.getRequestParameterMap();
        for (final String key : params.keySet()) {
            if (!PARAM_LIST.contains(key)) {
                String[] values = request.getParameterValues(key);
                if (values.length == 1) {
                    requestParams.put(key, values[0]);
                } else {
                    requestParams.put(key, values);
                }
            }
        }
        final U operation = getUpdateOperation();
        final Session session = resolver.adaptTo(Session.class);
        performBeforeActions(operation, session, request.getResource(), requestParams);

        SocialGraph graph = resolver.adaptTo(SocialGraph.class);
        Vertex userNode = graph.getVertex(this.userId);
        Vertex subscribedNode = graph.getVertex(this.subscribedId);
        final boolean isFollowedUser =
            subscribedId.startsWith(ClientUtilities.USER_ROOTPATH) || !subscribedId.startsWith("/");
        final String relType = isFollowedUser ? "USER" : "RESOURCE";
        final List<Relationship> subscriptions = new ArrayList<Relationship>();
        if (userNode != null && subscribedNode != null) {
            for (int i = 0; i < types.length; i++) {
                Relationship rel = userNode.getRelationship(Direction.OUTGOING, subscribedNode, types[i]);
                if (rel == null && states[i] == true) {
                    // not subscribed and want to subscribe
                    rel = userNode.createRelationshipTo(subscribedNode, types[i], relType);
                    subscriptions.add(rel);
                    LOG.info("Adding relationship user:{} node:{} types:{}", new Object[]{userNode.getId(),
                        subscribedNode.getId(), types[i]});
                } else if (rel != null && states[i] == false) {
                    rel.delete();
                    LOG.info("Remove relationship user:{}, endNode:{} type:{}", new Object[]{
                        rel.getStartNode().getId(), rel.getEndNode().getId(), types[i]});
                } else {
                    LOG.info("same state requestState:{} currentState:{} user:{} subscribed:{}", new Object[]{
                        states[i], (rel != null), userNode.getId(), subscribedNode.getId()});
                    if (rel != null) {
                        subscriptions.add(rel);
                    }
                }
            }

        }
        graph.save();
        SocialComponent component = getSocialComponent(request, subscriptions);
        if (component != null) {
            try {
                if (LOG.isInfoEnabled()) {
                    StringBuffer logInfo = new StringBuffer("\nRequest States: [");
                    for (int i = 0; i < types.length; i++) {
                        logInfo = logInfo.append(types[i]).append(":").append(states[i]).append(" ");
                    }
                    logInfo = logInfo.append("] Component:").append(component.toJSONString(true));
                    LOG.info(logInfo.toString());
                }
            } catch (Exception e) {
            }
            performAfterActions(operation, session, (SubscriptionCollection) component, requestParams);
        } else {
            LOG.warn("Failed to obtain the social component");
        }
        return component;
    }

    private SocialComponent getSocialComponent(final SlingHttpServletRequest request,
        final List<Relationship> subscriptions) {
        Resource resource = request.getResource();
        if (ResourceUtil.isNonExistingResource(resource)) {
            resource =
                new SyntheticResource(resource.getResourceResolver(),
                    "social/subscriptions/components/hbs/subscriptions", getResourceType());
        }
        final SocialComponentFactory factory = componentFactoryManager.getSocialComponentFactory(resource);
        if (factory instanceof SubscriptionCollectionComponentFactory) {
            return ((SubscriptionCollectionComponentFactory) factory).getSocialComponent(resource, request,
                subscriptions);
        } else {
            return (factory != null) ? factory.getSocialComponent(resource, request) : null;
        }
    }

    protected String getResourceType() {
        return SubscriptionCollection.RESOURCE_TYPE;
    }

    protected abstract U getUpdateOperation();

    protected abstract ResourceResolver getPrivilegedResolver() throws LoginException;
}
