/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.tally;

import java.util.Iterator;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;

import com.adobe.cq.social.tally.client.api.Response;
import com.adobe.cq.social.tally.client.api.ResponseValue;
import com.adobe.cq.social.tally.client.api.TallyException;

/**
 * A tally is a resource that can keep count or tally of a set of user responses on a specific target resource.
 * @param <T> The response value object that a tally will keep track of
 */
public interface Tally<T extends ResponseValue> {
    /**
     * @return the name of this tally
     */
    public String getName();

    /**
     * @return count of number of users who responded to this tally
     */
    public Long getTotalNumberOfResponses();

    /**
     * @param userId the user for whom the response is requested for
     * @return the response of the current user. null if the user has not responded
     * @throws TallyException when there was an error in reading the users latest response
     * @throws RepositoryException when there is a JCR error in reading the users latest response
     */
    public Response<T> getUserResponse(String userId) throws TallyException, RepositoryException;

    /**
     * @return the response of the current user. null if the user has not responded
     * @throws TallyException when there was an error in reading the users latest response
     */
    public T getCurrentUserResponse() throws TallyException;

    /**
     * @param start the beginning index which to return the responses from
     * @param numberOfResponses the number of responses to be returned
     * @return a list of responses
     */
    public Iterator<Response<T>> getResponses(Long start, int numberOfResponses);

    /**
     * @param start the beginning index which to return the responses from
     * @return a list of responses
     */
    public Iterator<Response<T>> getResponses(Long start);

    /**
     * @return a map of each response value to the count of number of users who have chosen the value. Example:
     *         [Helpful -&gt; 4, Unhelpful -&gt;5], [5-stars -&gt; 3, 4-stars -&gt; 2, 1-star -&gt; 4]
     */
    public Map<T, Long> getResponseTallies();

    /**
     * @return the resource for which this tally is being maintained
     */
    public Resource getTallyTarget();

    /**
     * sets the user response for this tally in jcr.
     * @param userId the authorizable id of the user who responded to the tally
     * @param responseValue the response value entered by the user for the tally
     * @throws TallyException when there is an error in setting the user response
     * @throws RepositoryException when there is a JCR error in setting the user response
     * @return the <code>Resource</code> object representing the response
     */
    public Resource setUserResponse(String userId, String responseValue) throws TallyException, RepositoryException;

    /**
     * sets the user response for this tally in jcr.
     * @param userId the authorizable id of the user who responded to the tally
     * @param responseValue the response value entered by the user for the tally
     * @param additionalPoperties any additional properties to write to the resource; may be null
     * @throws TallyException when there is an error in setting the user response
     * @throws RepositoryException when there is a JCR error in setting the user response
     * @return the <code>Resource</code> object representing the response
     */
    public Resource setUserResponse(String userId, String responseValue, Map<String, Object> additionalPoperties)
        throws TallyException, RepositoryException;

    /**
     * unsets the user response in jcr. Theoretically, this is like deleting the user response
     * @param userId the authorizable id of the user who responded to the tally
     * @throws TallyException when there is an error in trying to unset the user response
     * @throws RepositoryException when there is a JCR error in setting the user response
     * @return the <code>Resource</code> object representing the node that indicates an unset response
     */
    public Resource unsetUserResponse(String userId) throws TallyException, RepositoryException;

    /**
     * checks if the current logged in user has privileges to respond to this tally (based on ACLs).
     * @return true if user has privileges, false otherwise or if no user is logged in
     */
    public boolean canUserRespond();

    /**
     * checks if the current logged in user has privileges to respond to this tally (based on ACLs). This can track
     * the client context.
     * @param request the sling request to use as the basis of the user determination
     * @return true if user has privileges, false otherwise or if no user is logged in
     */
    public boolean canUserRespond(SlingHttpServletRequest request);

    /**
     * @param resourceType - the resource type to use when creating tally component resource nodes if needed in ugc.
     */
    void setTallyResourceType(String resourceType);

    /**
     * Check if the tally is using referrerUrl for its
     * {@link com.adobe.cq.social.tally.client.api.AbstractTally#getFriendlyUrl} or not.
     * @return
     */
    boolean getUseReferrerUrl();

}
