/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.tally.client.api;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.osgi.service.event.Event;

import com.adobe.cq.social.scf.core.SocialEvent;
import com.adobe.granite.activitystreams.ObjectTypes;
import com.adobe.granite.activitystreams.Verbs;

/**
 * Defines events that are triggered in tally components.
 */
public class TallyEvent extends SocialEvent<TallyEvent.TallyActions> {

    private static final long serialVersionUID = 1L;
    protected static final String PARENT_PATH = "PARENT_PATH";
    /**
     * The event topic suffix for tally events
     */
    public static final String TALLY_TOPIC_PREFIX = "tally/";

    /**
     * @param tallyPath - the jcr node path to the tally resource on which this event was triggered.
     * @param userId - the user id of the user who triggered the event.
     * @param parentPath - the jcr node path to the parent node of the tally resource.
     * @param tallyType - the type of tally component (voting, rating, etc,.).
     * @param action - the tally action that triggered this event.
     */
    public TallyEvent(final String tallyPath, final String userId, final String parentPath, final String tallyType,
        final TallyEvent.TallyActions action) {
        this(tallyPath, userId, parentPath, tallyType, action, null, null);
    }

    /**
     * @param tallyPath - the jcr node path to the tally resource on which this event was triggered.
     * @param userId - the user id of the user who triggered the event.
     * @param parentPath - the jcr node path to the parent node of the tally resource.
     * @param tallyType - the type of tally component (voting, rating, etc,.).
     * @param action - the tally action that triggered this event.
     */
    public TallyEvent(final String tallyPath, final String userId, final String parentPath, final String tallyType,
        final TallyEvent.TallyActions action, final Map<String, Object> objectExtraProps,
        final Map<String, Object> targetExtraProps) {
        super(TALLY_TOPIC_PREFIX + tallyType, tallyPath, userId, action.setType(tallyType), new BaseEventObject(
            tallyPath, ObjectTypes.REVIEW, objectExtraProps), new BaseEventObject(parentPath, ObjectTypes.ARTICLE,
            targetExtraProps), new HashMap<String, Object>(1) {
            private static final long serialVersionUID = 1L;
            {
                if (parentPath != null) {
                    this.put(PARENT_PATH, parentPath);
                }

            }
        });
    }

    private TallyEvent(final Event event) {
        super(event);
    }

    /**
     * List of available tally actions that can trigger a tally event.
     */
    public static enum TallyActions implements SocialEvent.SocialActions {
        SET_RESPONSE, UNSET_RESPONSE;
        String type;

        TallyActions setType(final String type) {
            this.type = type.toLowerCase();
            return this;
        }

        @Override
        public String getVerb() {
            switch (this) {
                case SET_RESPONSE:
                    return Verbs.ADD + " " + type;
                case UNSET_RESPONSE:
                    return Verbs.REMOVE + " " + type;
                default:
                    throw new IllegalArgumentException("Unsupported action");
            }
        }
    }

    /**
     * @return the jcr node path to the parent of the tally resource on which this event was triggered.
     */
    public String getParentPath() {
        return (String) this.getProperty(PARENT_PATH);
    }

    /**
     * Retrieves the tally event from the given generic {@link Event}.
     * @param event The event.
     * @return The tally event or <code>null</code> if the given event is not a tally event.
     */
    public static TallyEvent fromEvent(final Event event) {
        if (StringUtils.startsWith(event.getTopic(), SocialEvent.SOCIAL_EVENT_TOPIC_PREFIX + TALLY_TOPIC_PREFIX)) {
            return new TallyEvent(event);
        }
        return null;
    }

    /**
     * @return the tally component type of the tally resource on which this event was triggered.
     */
    public String getTallyType() {
        return this.getTopic().substring(this.getTopic().lastIndexOf("/") + 1);
    }
}
