/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

import java.util.Collections;
import java.util.List;
import java.util.Vector;

/**
 * FullTextConstraint restricts the results to a specific full text expression.
 */
public class FullTextConstraint extends AbstractConstraint implements Constraint {

    /**
     * Full text search expression to match.
     */
    private String expression;

    /**
     * A list of properties which should be included in the search. If no properties are specified all properties will
     * be included.
     */
    private List<String> includedProperties;

    /**
     * Creates a new instance of FullTextConstraint with the given expression.
     * @param expression String containing the full text expression.
     */
    public FullTextConstraint(final String expression) {
        super(Operator.And);
        this.expression = expression;
    }

    /**
     * Creates a new instance of FullTextConstraint with the given expression.
     * @param expression String containing the full text expression.
     * @param propertyName String containing the name of a property to include in the search.
     */
    public FullTextConstraint(final String expression, final String propertyName) {
        super(Operator.And);
        this.expression = expression;
        this.includeProperty(propertyName);
    }

    /**
     * Gets whether or not the constraint defines which properties should be included in the full text search.
     * @return True if this constraint defines the properties to include, false otherwise.
     */
    public boolean definesPropertiesToInclude() {
        return null != this.includedProperties && this.includedProperties.size() > 0;
    }

    /**
     * Adds a property to include in the full text search.
     * @param propertyName String containing the name of the property to include.
     */
    public void includeProperty(final String propertyName) {
        if (null == this.includedProperties) {
            this.includedProperties = new Vector<String>();
        }
        this.includedProperties.add(propertyName);
    }

    /**
     * Gets the full text expression to match.
     * @return String containing the full text expression.
     */
    public String getExpression() {
        return this.expression;
    }

    /**
     * Sets the full text expression to match.
     * @param expression String containing the full text expression to match.
     */
    public void setExpression(final String expression) {
        this.expression = expression;
    }

    /**
     * Gets the list of included properties.
     * @return the list of included properties.
     */
    public List<String> getIncludedProperties() {
        if (null == this.includedProperties) {
            return Collections.<String>emptyList();
        }
        return this.includedProperties;
    }

    /**
     * Visit this constraint.
     * @param constraintVisitor the visitor for this constraint.
     */
    @Override
    public void accept(final ConstraintVisitor constraintVisitor) {
        constraintVisitor.visitFullTextConstraint(this);
    }
}
