/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

/**
 * RangeConstraint is a property constraint which restricts the value of a property to a specific range.
 * @param <T> the type of object that is being queried.
 */
public class RangeConstraint<T> extends AbstractPropertyConstraint implements Constraint {

    /**
     * The min value.
     */
    private T minValue;

    /**
     * The max value.
     */
    private T maxValue;

    /**
     * True if range is inclusive.
     */
    private boolean inclusive;

    /**
     * Creates a new instance of RangeConstraint (inclusive) against the given propertyName with minValue and
     * maxValue.
     * @param propertyName Name of the property being constrained.
     * @param minValue Minimum value for the range.
     * @param maxValue Maximum value for the range.
     */
    public RangeConstraint(final String propertyName, final T minValue, final T maxValue) {
        this(propertyName, minValue, maxValue, true);
    }

    /**
     * Creates a new instance of RangeConstraint against the given propertyName with minValue and maxValue.
     * @param propertyName Name of the property being constrained.
     * @param minValue Minimum value for the range.
     * @param maxValue Maximum value for the range.
     * @param inclusive True if the results should include minValue and maxValue, false otherwise.
     */
    public RangeConstraint(final String propertyName, final T minValue, final T maxValue, final boolean inclusive) {
        this(propertyName, minValue, maxValue, inclusive, Operator.And);
    }

    /**
     * Creates a new instance of RangeConstraint against the given propertyName with minValue and maxValue.
     * @param propertyName Name of the property being constrained.
     * @param minValue Minimum value for the range.
     * @param maxValue Maximum value for the range.
     * @param inclusive True if the results should include minValue and maxValue, false otherwise.
     * @param operator Operator that defines how this constraint is joined with other constraints.
     */
    public RangeConstraint(final String propertyName, final T minValue, final T maxValue, final boolean inclusive,
        final Operator operator) {
        super(propertyName, operator);
        this.minValue = minValue;
        this.maxValue = maxValue;
        this.inclusive = inclusive;
    }

    /**
     * Gets the minimum value of this constraint.
     * @return The minimum value of this constraint.
     */
    public T getMinValue() {
        return this.minValue;
    }

    /**
     * Sets the minimum value of this constraint.
     * @param minValue The minimum value of this constraint.
     */
    public void setMinValue(final T minValue) {
        this.minValue = minValue;
    }

    /**
     * Gets the maximum value of this constraint.
     * @return The maximum value of this constraint.
     */
    public T getMaxValue() {
        return this.maxValue;
    }

    /**
     * Gets the maximum value of this constraint.
     * @param maxValue The minimum value of this constraint.
     */
    public void setMaxValue(final T maxValue) {
        this.maxValue = maxValue;
    }

    /**
     * Whether the range should include the minimum value and maximum value (inclusive) or not.
     * @return True if the range is inclusive, false otherwise.
     */
    public boolean isInclusive() {
        return this.inclusive;
    }

    /**
     * Visit this range constraint.
     * @param constraintVisitor the ConstraintVisitor.
     */
    @Override
    public void accept(final ConstraintVisitor constraintVisitor) {
        constraintVisitor.visitRangeConstraint(this);
    }
}
