/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import com.adobe.cq.social.srp.FacetRangeField;
import com.adobe.cq.social.srp.FacetSearchResult;

/**
 * Interface for performing related search and faceted search.
 */
public interface RelatedSearch {

    /**
     * Return a list of related elements, given a query.
     * @param scName The name of the search collection.
     * @param query The string to search for.
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param statusFilter The status to filter on.
     * @param mltFields Node properties to search on.
     * @param maxResults The maximum results returned.
     * @param minTermFreq The frequency which terms will be ignored in the source doc
     * @param minDocFreq The frequency at which words will be ignored which do not occur in at least this many docs.
     * @return The list of matching documents.
     * @throws RepositoryException thrown when there are issues with repository.
     */
    List<String> findMLT(String scName, String query, String statusFilter, String resourceTypeFilter,
        String componentFilter, String[] mltFields, int maxResults, int minTermFreq, int minDocFreq)
        throws RepositoryException;

    /**
     * Return a list of related elements, given a query.
     * @param scName The name of the search collection.
     * @param query The string to search for.
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param mltFields Node properties to search on.
     * @param maxResults The maximum results returned.
     * @param minTermFreq The frequency which terms will be ignored in the source doc
     * @param minDocFreq The frequency at which words will be ignored which do not occur in at least this many docs.
     * @return The list of matching documents.
     * @throws RepositoryException thrown when there are issues with repository.
     */
    List<String> findMLT(String scName, String query, String resourceTypeFilter, String componentFilter,
        String[] mltFields, int maxResults, int minTermFreq, int minDocFreq) throws RepositoryException;

    /**
     * Return a list of categories [Facets] and their counts for a field.
     * @param scName The name of the search collection.
     * @param fieldName The field name to be retrieved [Ex: Author, CategoryDate, cq:Tags]
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @return The map of Categories[values] and their counts.
     * @throws RepositoryException thrown when there are issues with repository.
     */
    Map<String, Integer> findFacets(String scName, String fieldName, String resourceTypeFilter,
        String componentFilter, int count) throws RepositoryException;

    /**
     * Return a list of categories [Facets] and their counts for multiple fields. Takes a list of field names for
     * which facet counts are wanted. Avoids multiple trips to the indexing engine, which could be in the cloud and
     * have a non negligible network latency
     * @param scName The name of the search collection.
     * @param fieldNames The field names to be retrieved [Ex: Author, CategoryDate, cq:Tags]
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @return The map of names and facet results for each field: Map<String, Map<String, Integer>>
     * @throws RepositoryException thrown when there are issues with repository.
     * @deprecated Use method without the scName argument.
     */
    @Deprecated
    Map<String, Map<String, Integer>> findFacets(String scName, List<String> fieldNames, String resourceTypeFilter,
        String componentFilter, int count) throws RepositoryException;

    /**
     * Return a list of categories [Facets] and their counts for multiple fields. Takes a list of field names for
     * which facet counts are wanted. Avoids multiple trips to the indexing engine, which could be in the cloud and
     * have a non negligible network latency.
     * @param fieldNames The field names to be retrieved [Ex: Author, CategoryDate, cq:Tags]
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @return The map of names and facet results for each field: Map<String, Map<String, Integer>>
     * @throws RepositoryException thrown when there are issues with repository.
     */
    Map<String, Map<String, Integer>> findFacets(List<String> fieldNames, String resourceTypeFilter,
        String componentFilter, int count) throws RepositoryException;

    /**
     * Return a list of categories [Facets] and their counts for multiple fields. Takes a list of field names for
     * which facet counts are wanted. Avoids multiple trips to the indexing engine, which could be in the cloud and
     * have a non negligible network latency.
     * @param fieldNames The field names to be retrieved [Ex: Author, CategoryDate, cq:Tags]
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @param isVisibleOnly Should include drafts and spam posts etc
     * @param includeChildren Should Subpaths be included
     * @return The map of names and facet results for each field: Map<String, Map<String, Integer>>
     * @throws RepositoryException thrown when there are issues with repository.
     */

    Map<String, Map<String, Integer>> findFacets(List<String> fieldNames, String resourceTypeFilter,
        String componentFilter, int count, boolean isVisbleOny, boolean includeChildren) throws RepositoryException;

    /**
     * Return a list of categories [Facets] and their counts for multiple fields. Takes a list of field names for
     * which facet counts are wanted. Avoids multiple trips to the indexing engine, which could be in the cloud and
     * have a non negligible network latency.
     * @param fieldNames The field names to be retrieved [Ex: Author, CategoryDate, cq:Tags]
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @param isVisibleOnly Should Subpaths be included
     * @return The map of names and facet results for each field: Map<String, Map<String, Integer>>
     * @throws RepositoryException thrown when there are issues with repository.
     */
    Map<String, Map<String, Integer>> findFacets(List<String> fieldNames, String resourceTypeFilter,
        String componentFilter, int count, boolean isVisibleOnly) throws RepositoryException;

    /**
     * Return a list of categories [Facets] grouped by the range gap.
     * @param facetRanges The facet range (field, start, end, gap)
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @return The map of names and facet range results for each field: Map<String, Map<String, Integer>>
     * @throws RepositoryException thrown when there are issues with repository.
     */
    Map<String, Map<String, Integer>> findFacetRanges(List<FacetRangeField> facetRanges, String resourceTypeFilter,
        String componentFilter, int count) throws RepositoryException;

    /**
     * Return a list of categories [Facets] and their counts and ranges grouped by the range gap.
     * @param fieldNames The field names for the counts [Ex: Author, CategoryDate, cq:Tags]
     * @param facetRanges The facet range (field, start, end, gap)
     * @param resourceTypeFilter The resource type to filter on.
     * @param componentFilter The path of the component to filter on.
     * @param count The maximum results returned.
     * @return Count and range maps in the FacetSearchResult.
     * @throws RepositoryException thrown when there are issues with repository.
     */
    FacetSearchResult findFacets(List<String> fieldNames, List<FacetRangeField> facetRanges,
        String resourceTypeFilter, String componentFilter, int count) throws RepositoryException;
}
