/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * SetConstraint is a property constraint which restricts the value of a property to one of a set of values.
 * @param <T> the type of objects in the set.
 */
public class SetConstraint<T> extends AbstractPropertyConstraint implements Constraint {

    /**
     * The values the property value is being constrained to.
     */
    private List<T> values;

    /**
     * Creates a new SetConstraint.
     * @param propertyName Name of the property being constrained.
     */
    public SetConstraint(final String propertyName) {
        this(propertyName, Operator.And);
    }

    /**
     * Creates a new SetConstraint against the provided property name with all the given values.
     * @param propertyName Name of the property being constrained.
     * @param values Collection of values to constrain the property to.
     */
    public SetConstraint(final String propertyName, final Collection<T> values) {
        this(propertyName, values, Operator.And);
    }

    /**
     * Creates a new SetConstraint against the provided property name.
     * @param propertyName Name of the property being constrained.
     * @param operator Operator that defines how this constraint is joined with other constraints.
     */
    public SetConstraint(final String propertyName, final Operator operator) {
        super(propertyName, operator);
    }

    /**
     * Creates a new SetConstraint against the provided property name with all the given values.
     * @param propertyName Name of the property being constrained.
     * @param values Collection of values to constrain the property to.
     * @param operator Operator that defines how this constraint is joined with other constraints.
     */
    public SetConstraint(final String propertyName, final Collection<T> values, final Operator operator) {
        super(propertyName, operator);
        this.addAll(values);
    }

    /**
     * Adds a new value to this set constraint.
     * @param value Value to add.
     */
    public void add(final T value) {
        if (null == this.values) {
            this.values = new ArrayList<T>();
        }
        this.values.add(value);
    }

    /**
     * Adds all values to the set.
     * @param vals Collection of values to add.
     */
    public void addAll(final Collection<T> vals) {
        if (null == this.values) {
            this.values = new ArrayList<T>();
        }
        this.values.addAll(vals);
    }

    /**
     * Gets the number of values in this set.
     * @return int containing the number of values in this set.
     */
    public int size() {
        int result;
        if (null == this.values) {
            result = 0;
        } else {
            result = this.values.size();
        }
        return result;
    }

    /**
     * Gets the value at the specific location.
     * @param index Index of the value to get.
     * @return Value at the given location.
     */
    public T get(final int index) {
        T result;
        if (null == this.values) {
            throw new IndexOutOfBoundsException("No values exist cannot retrieve value for index " + index);
        } else {
            result = this.values.get(index);
        }
        return result;
    }

    /**
     * Sets the value at the specified index.
     * @param i index of the value to set.
     * @param value value to set.
     */
    public void set(final int i, final T value) {
        if (null == this.values || i >= this.values.size()) {
            this.add(value);
        } else if (i < 0) {
            throw new IndexOutOfBoundsException(i + " is not in range [0, " + (this.values.size() - 1) + "]");
        }
        this.values.set(i, value);
    }

    /**
     * Gets the all the values in this set.
     * @return Collection containing all the values of this set.
     */
    public Collection<T> getValues() {
        Collection<T> result;
        if (null == this.values) {
            result = Collections.emptyList();
        } else {
            result = this.values;
        }
        return result;
    }

    /**
     * Visit this range constraint.
     * @param constraintVisitor the ConstraintVisitor.
     */
    @Override
    public void accept(final ConstraintVisitor constraintVisitor) {
        constraintVisitor.visitSetConstraint(this);
    }
}
