/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.lucene.queryParser.QueryParser;

import com.adobe.cq.social.ugc.api.UgcSort.Direction;

/**
 * The UGC filter.
 */
public class UgcFilter {

    /**
     * The default content type.
     */
    public static final String DEFAULT_CONTENT_TYPE = "cq:Comment";

    /**
     * The content type for this filter.
     */
    private String contentType;

    /**
     * The sort order list.
     */
    private List<UgcSort> sortOrder;

    /**
     * The collection of constraints.
     */
    private Collection<Constraint> constraints = new Vector<Constraint>();

    /**
     * The map of variable names and values.
     */
    private Map<String, Object> variables = new HashMap<String, Object>();

    // these support backwards compatibility

    /**
     * The path filters ConstraintGroup.
     */
    private ConstraintGroup pathFilters;

    /**
     * The map of or filter ConstraintGroups.
     */
    private Map<String, ConstraintGroup> orFilters;

    /**
     * The map of and filter ConstraintGroups.
     */
    private Map<String, ConstraintGroup> andFilters;

    /**
     * Creates a new UgcFilter with the default content type.
     */
    public UgcFilter() {
        this(DEFAULT_CONTENT_TYPE);
    }

    /**
     * Creates a new UgcFilter with the given content type.
     * @param contentType String containing the given content type.
     */
    public UgcFilter(final String contentType) {
        super();
        this.contentType = contentType;
    }

    /**
     * Adds a constraint to this UgcFilter.
     * @param constraint Constraint to add.
     */
    public void addConstraint(final Constraint constraint) {
        if (null == this.constraints) {
            this.constraints = new Vector<Constraint>();
        }
        this.constraints.add(constraint);
    }

    /**
     * A Convenience method which will set the operator of the given constraint to AND and add it to this UgcFilter.
     * @param constraint Constraint to add.
     */
    public void and(final Constraint constraint) {
        constraint.setOperator(Operator.And);
        addConstraint(constraint);
    }

    /**
     * A Convenience method which will set the operator of the given constraint to OR and add it to this UgcFilter.
     * @param constraint Constraint to add.
     */
    public void or(final Constraint constraint) {
        constraint.setOperator(Operator.Or);
        addConstraint(constraint);
    }

    /**
     * Gets whether or not the UgcFilter has constraints.
     * @return True if the UgcFilter has constraints, false otherwise.
     */
    public boolean hasConstraints() {
        return null != this.constraints;
    }

    /**
     * Gets all constraints which have been added to this UgcFilter. An empty collection is returned if no constraints
     * have been added.
     * @return A Collection containing all constraints that have been added to this UgcFilter.
     */
    public Collection<Constraint> getConstraints() {
        Collection<Constraint> result;
        if (null == this.constraints) {
            result = Collections.<Constraint>emptyList();
        } else {
            result = this.constraints;
        }
        return result;
    }

    /**
     * Add a UgcSort to the list.
     * @param sort the UgcSort to add.
     */
    public void addSort(final UgcSort sort) {
        if (null == this.sortOrder) {
            this.sortOrder = new Vector<UgcSort>();
        }
        this.sortOrder.add(sort);
    }

    /**
     * Returns the content type for this UgcFilter. This is the nodetype and is only useful with JSRP. For cross-SRP
     * code, you should use a ValueConstraint with the sling:resourceType
     * @return String containing the content type for this UgcFilter.
     */
    public String getContentType() {
        return this.contentType;
    }

    /**
     * Set the node type to be matched. This is only useful with JSRP. For cross-SRP code, you should use a
     * ValueConstraint with the sling:resourceType
     * @param contentType the content type to use
     */
    public void setContentType(final String contentType) {
        this.contentType = contentType;
    }

    /**
     * Escape any special characters in the specified string by a preceding \.
     * @param str the input string
     * @return a String where those characters that QueryParser expects to be escaped are escaped by a preceding \
     */
    public static String escape(final String str) {
        return QueryParser.escape(str);
    }

    /**
     * Gets whether or not this UgcFilter is sorted.
     * @return True if this UgcFilter is sorted, false otherwise.
     */
    public boolean isSorted() {
        return null != this.sortOrder && 0 < this.sortOrder.size();
    }

    /**
     * Gets the sort order of this UgcFilter.
     * @return A List of UgcSort objects or null if the UgcFilter is not sorted.
     */
    public List<UgcSort> getSortOrder() {
        return this.sortOrder;
    }

    /**
     * Gets whether or not variables have been added to this UgcFilter.
     * @return True if the UgcFilter has a variable, false otherwise.
     */
    public boolean hasVariables() {
        return null != this.variables && 0 < this.variables.size();
    }

    /**
     * Set a variable.
     * @param variableName the name of the variable.
     * @param variable the value of the variable.
     * @param <T> the type of variable.
     */
    public <T> void setVariableValue(final String variableName, final T variable) {
        if (null == this.variables) {
            this.variables = new HashMap<String, Object>();
        }
        this.variables.put(variableName, variable);
    }

    /**
     * Return the variable map.
     * @return the variable map.
     */
    public Map<String, Object> getVariables() {
        Map<String, Object> result;
        if (null == this.variables) {
            result = Collections.<String, Object>emptyMap();
        } else {
            result = this.variables;
        }
        return result;
    }

    /**
     * Accept the constraint visitor.
     * @param constraintVisitor the ConstraintVisitor.
     */
    public void accept(final ConstraintVisitor constraintVisitor) {
        if (this.hasConstraints()) {
            for (final Constraint constraint : this.constraints) {
                constraint.accept(constraintVisitor);
            }
        }
    }

    /**
     * Creates constraint groups which filterAndByProperty and filterOrByProperty will add constraints to. These
     * constraint groups are used to maintain backwards compatibility with these deprecated apis.
     */
    protected void initConstraintGroups() {
        if (null == this.orFilters) {
            this.orFilters = new HashMap<String, ConstraintGroup>();
            this.andFilters = new HashMap<String, ConstraintGroup>();
        }
    }

    /**
     * Gets a ComparisonType which matches the given comparison.
     * @param comparison comparison.
     * @return ComparisonType which matches the given comparison.
     */
    protected ComparisonType getComparisonType(final Comparison comparison) {
        ComparisonType comparisonType;
        switch (comparison) {
            case EQUALS:
                comparisonType = ComparisonType.Equals;
                break;
            case GREATER_THAN:
                comparisonType = ComparisonType.GreaterThan;
                break;
            case GREATER_THAN_EQUAL_TO:
                comparisonType = ComparisonType.GreaterThanOrEqualTo;
                break;
            case LESS_THAN:
                comparisonType = ComparisonType.LessThan;
                break;
            case LESS_THAN_EQUAL_TO:
                comparisonType = ComparisonType.LessThanOrEqualTo;
                break;
            case NOT_EQUALS:
                comparisonType = ComparisonType.NotEquals;
                break;
            default:
                throw new RuntimeException("Comparison " + comparison.name() + " not supported");
        }
        return comparisonType;
    }

    /**
     * @deprecated Use the {@link #getSortOrder() getSortOrder} method.
     * @return the UgcSort.
     */
    @Deprecated
    public UgcSort getSort() {
        if (isSorted()) {
            return this.sortOrder.get(0);
        }
        return null;
    }

    /**
     * Due to a bug and for backwards compatibility reasons this method actually adds a descending sort.
     * {@link #addSort(UgcSort) addSort} should be used instead.
     * @param propertyName Name of the property to sort.
     * @deprecated Use the {@link #addSort(UgcSort) addSort} method.
     */
    @Deprecated
    public void sortByAscending(final String propertyName) {
        this.addSort(new UgcSort(propertyName, Direction.Asc));
    }

    /**
     * Due to a bug and for backwards compatibility reasons this method actually adds a ascending sort.
     * {@link #addSort(UgcSort) addSort} should be used instead.
     * @param propertyName Name of the property to sort.
     * @deprecated Use the {@link #addSort(UgcSort) addSort} method.
     */
    @Deprecated
    public void sortByDescending(final String propertyName) {
        this.addSort(new UgcSort(propertyName, Direction.Desc));
    }

    /**
     * @deprecated Use {@link #or(Constraint) or} with a {@link ValueConstraint ValueConstraint} instead.
     * @param propertyName the name of the property.
     * @param propertyValue the property value.
     * @param comparison the type of comparison.
     */
    @Deprecated
    public void filterOrByProperty(final String propertyName, final String propertyValue, final Comparison comparison) {
        filterOrByConstraintGroup(propertyName, propertyName, propertyValue, comparison);
    }

    /**
     * @deprecated Use {@link #or(Constraint) or} with a {@link ValueConstraint ValueConstraint} instead.
     * @param constraintGroupName the name of the constraint group.
     * @param propertyName the name of the property.
     * @param propertyValue the property value.
     * @param comparison the type of comparison.
     */
    @Deprecated
    public void filterOrByConstraintGroup(final String constraintGroupName, final String propertyName,
        final String propertyValue, final Comparison comparison) {
        initConstraintGroups();

        if (!orFilters.containsKey(constraintGroupName)) {
            final ConstraintGroup cg = new ConstraintGroup();
            and(cg);
            orFilters.put(constraintGroupName, cg);
        }

        final ComparisonType comparisonType = getComparisonType(comparison);

        orFilters.get(constraintGroupName).addConstraint(
            new ValueConstraint(propertyName, propertyValue, comparisonType, Operator.Or));
    }

    /**
     * @deprecated Use {@link #and(Constraint) and} with a {@link ValueConstraint ValueConstraint} instead.
     * @param propertyName the name of the property.
     * @param propertyValue the property value.
     * @param comparison the type of comparison.
     */
    @Deprecated
    public void filterAndByProperty(final String propertyName, final String propertyValue, final Comparison comparison) {
        filterAndByConstraintGroup(propertyName, propertyName, propertyValue, comparison);
    }

    /**
     * @deprecated Use {@link #and(Constraint) and} with a {@link ValueConstraint ValueConstraint} instead.
     * @param constraintGroupName the name of the constraint group.
     * @param propertyName the name of the property.
     * @param propertyValue the property value.
     * @param comparison the type of comparison.
     */
    @Deprecated
    public void filterAndByConstraintGroup(final String constraintGroupName, final String propertyName,
        final String propertyValue, final Comparison comparison) {
        initConstraintGroups();

        if (!andFilters.containsKey(constraintGroupName)) {
            final ConstraintGroup cg = new ConstraintGroup();
            and(cg);
            andFilters.put(constraintGroupName, cg);
        }

        final ComparisonType comparisonType = getComparisonType(comparison);
        andFilters.get(constraintGroupName).addConstraint(
            new ValueConstraint(propertyName, propertyValue, comparisonType, Operator.And));
    }

    /**
     * @deprecated Use {@link #hasConstraints hasConstraints} instead.
     * @return true if there are property filters.
     */
    @Deprecated
    public boolean hasPropertyFilters() {
        return false;
    }

    /**
     * @deprecated Use {@link #getConstraints getConstraints} instead.
     * @return the map of OR property filters.
     */
    @Deprecated
    public Map<String, List<PropertyFilter>> getOrPropertyFilters() {
        return Collections.<String, List<PropertyFilter>>emptyMap();
    }

    /**
     * @deprecated Use {@link #getConstraints getConstraints} instead.
     * @return the map of AND property filters.
     */
    @Deprecated
    public Map<String, List<PropertyFilter>> getAndPropertyFilters() {
        return Collections.<String, List<PropertyFilter>>emptyMap();
    }

    /**
     * @deprecated Use {@link #addConstraint(Constraint) addConstraint} with {@link PathConstraint PathConstraint}
     *             instead. Set the content type to filter.
     * @param ct the content type to filter.
     */
    @Deprecated
    public void filterContentType(final String ct) {
        // TODO JH add content type constraint?
        this.contentType = ct;
    }

    /**
     * @deprecated Use {@link #getConstraints()} getConstraints} instead.
     * @return true if content type is filtered.
     */
    @Deprecated
    public boolean isContentTypeFiltered() {
        return false;
    }

    /**
     * @deprecated Use {@link #getContentType() getContentType} instead.
     * @return the content type filter.
     */
    @Deprecated
    public String getContentFilter() {
        return null;
    }

    /**
     * @deprecated Use {@link #addConstraint(Constraint) addConstraint} with {@link PathConstraint PathConstraint}
     *             instead.
     * @param parentPath the path of the parent to filter.
     */
    @Deprecated
    public void filterByPath(final String parentPath) {
        if (null == pathFilters) {
            this.pathFilters = new ConstraintGroup(Operator.And);
            addConstraint(this.pathFilters);
        }
        this.pathFilters.addConstraint(new PathConstraint(parentPath, PathConstraintType.IsDescendantNode,
            Operator.Or));
    }

    /**
     * @deprecated use {@link #hasConstraints() hasConstraints} instead.
     * @return true if the path is filtered.
     */
    @Deprecated
    public boolean isPathFiltered() {
        return false;
    }

    /**
     * @deprecated Use {@link #getConstraints()} getConstraints} instead.
     * @return the array of path filters.
     */
    @Deprecated
    public String[] getPathFilters() {
        return new String[0];
    }

    /**
     * The property filter.
     * @deprecated Use {@link #addConstraint(Constraint) addConstraint} instead of creating PropertyFilters.
     */
    @Deprecated
    public final class PropertyFilter {
        /**
         * The comparison.
         */
        private final Comparison comparison;

        /**
         * The value.
         */
        private final String value;

        /**
         * Construct a PropertyFilter.
         * @param v the value of the property.
         * @param c the type of comparison.
         */
        private PropertyFilter(final String v, final Comparison c) {
            value = v;
            comparison = c;
        }

        /**
         * Return the comparison.
         * @return the comparison.
         */
        public Comparison getComparison() {
            return comparison;
        }

        /**
         * Return the value.
         * @return the value.
         */
        public String getValue() {
            return value;
        }
    }

    /**
     * @deprecated Use {@link ComparisonType ComparisonType} instead.
     */
    @Deprecated
    public enum Comparison {

        /**
         * The equals comparison.
         */
        EQUALS("=", false),
        /**
         * The not equals comparison.
         */
        NOT_EQUALS("=", true),
        /**
         * The less than comparison.
         */
        LESS_THAN("<", false),
        /**
         * The greater than comparison.
         */
        GREATER_THAN(">", false),
        /**
         * The less than or equal to comparison.
         */
        LESS_THAN_EQUAL_TO("<=", false),
        /**
         * The greater than or equal to comparison.
         */
        GREATER_THAN_EQUAL_TO(">=", false);

        /**
         * The name of the comparison.
         */
        private String name;

        /**
         * True if negation.
         */
        private boolean negation;

        /**
         * Construction a comparison.
         * @param s the name.
         * @param n true if negation.
         */
        private Comparison(final String s, final boolean n) {
            name = s;
            negation = n;
        }

        /**
         * Return true if negation.
         * @return true if negation.
         */
        public boolean negation() {
            return negation;
        }

        /**
         * Return the name of the comparison.
         * @return the name of the comparison.
         */
        @Override
        public String toString() {
            return name;
        }
    }

}
