/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

import org.apache.lucene.document.Field;

/**
 * An index property specified by index handers.
 */
public class UgcIndexProperty {

    /**
     * The index types.
     */
    public enum Index {
        /**
         * Analyzed.
         */
        ANALYZED,
        /**
         * No analyzed.
         */
        NOT_ANALYZED,
        /**
         * Not analyzed and no norms.
         */
        NOT_ANALYZED_NO_NORMS,
        /**
         * Analyzed and no norms.
         */
        ANALYZED_NO_NORMS;
    }

    /**
     * The name of the property.
     */
    private final String name;

    /**
     * The value of the property.
     */
    private final Object value;

    /**
     * True if indexed with store set to true.
     */
    private final boolean store;

    /**
     * The index type.
     */
    private final Index index;

    /**
     * True if full text.
     */
    private final boolean fulltext;

    /**
     * Create a UgcIndexProperty by specifying its name, value and how it will be saved in the index.
     * @param name The name of the field
     * @param value The string to process
     * @param store Whether <code>value</code> should be stored in the index
     * @param index Whether the field should be indexed, and if so, if it should be tokenized before indexing
     */
    public UgcIndexProperty(final String name, final Object value, final boolean store, final Index index) {
        this(name, value, store, index, false);
    }

    /**
     * Create a UgcIndexProperty by specifying its name, value and how it will be saved in the index.
     * @param name The name of the field
     * @param value The string to process
     * @param store Whether <code>value</code> should be stored in the index
     * @param index Whether the field should be indexed, and if so, if it should be tokenized before indexing
     * @param fulltext true if full text.
     */
    public UgcIndexProperty(final String name, final Object value, final boolean store, final Index index,
        final boolean fulltext) {
        this.name = name;
        this.value = value;
        this.store = store;
        this.index = index;
        this.fulltext = fulltext;
    }

    /**
     * Return fulltext.
     * @return the value of fulltext.
     */
    public boolean isFullText() {
        return this.fulltext;
    }

    /**
     * Return property name.
     * @return property name.
     */
    public String getName() {
        return name;
    }

    /**
     * Return property value.
     * @return property value.
     */
    public Object getValue() {
        return value;
    }

    /**
     * Return true if store.
     * @return true if store.
     */
    public boolean isStore() {
        return store;
    }

    /**
     * Return the index type.
     * @return the index type.
     */
    public Index getIndex() {
        return index;
    }

    /**
     * Return the field index.
     * @return the field index.
     */
    public Field.Index getFieldIndex() {
        switch (index) {
            case ANALYZED:
                return Field.Index.ANALYZED;
            case NOT_ANALYZED:
                return Field.Index.NOT_ANALYZED;
            case ANALYZED_NO_NORMS:
                return Field.Index.ANALYZED_NO_NORMS;
            case NOT_ANALYZED_NO_NORMS:
                return Field.Index.NOT_ANALYZED_NO_NORMS;
            default:
                return null;
        }
    }
}
