/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.social.ugc.api;

import java.util.Date;

import org.apache.commons.lang.StringUtils;

import com.adobe.cq.social.sc.lucene.DateField;

/**
 * Indexer utility methods.
 */
public final class UgcIndexerUtil {

    /**
     * The PAD char.
     */
    private static final char PAD_CHAR = '0';

    /**
     * The negative prefix.
     */
    private static final String NEGATIVE_PREFIX = "-";

    /**
     * The positive prefix.
     */
    private static final String POSITIVE_PREFIX = "0";

    /**
     * The max integer padding.
     */
    private static final int INTEGER_MAX_PADDING = Integer.toString(Integer.MAX_VALUE).length();

    /**
     * The max long padding.
     */
    private static final int LONG_MAX_PADDING = Long.toString(Long.MAX_VALUE).length();

    /**
     * UgcIndexerUtil should not be constructed since it only has static utility methods.
     */
    private UgcIndexerUtil() {
    }

    /**
     * Convert date to string.
     * @param date the date to convert.
     * @return the string conversion.
     */
    public static String dateToString(final Date date) {
        return DateField.dateToString(date);
    }

    /**
     * Escape the given query.
     * @param query the query to escape.
     * @return the escaped query.
     */
    public static String escapeQuery(final String query) {
        if (query == null) {
            return null;
        }

        // Tried using QueryParser.escape here but it didn't work. To work
        // around, just remove key characters that would screw up the query.
        String escapedQuery = query.replace("/", "");
        escapedQuery = escapedQuery.replace(":", "");
        escapedQuery = escapedQuery.replace("_", "");
        escapedQuery = escapedQuery.replace("-", "");
        return escapedQuery;
    }

    /**
     * Convert long to string.
     * @param value the long to convert.
     * @return the string conversion.
     */
    public static String longToString(final long value) {
        return longToString(value, LONG_MAX_PADDING);
    }

    /**
     * Convert long to string.
     * @param value the long to convert.
     * @param padSize the pad size.
     * @return the string conversion.
     */
    public static String longToString(final long value, final int padSize) {
        return (value >= 0 ? POSITIVE_PREFIX : NEGATIVE_PREFIX)
                + StringUtils.leftPad(Long.toString(value), padSize, PAD_CHAR);
    }

    /**
     * Convert the int to a string.
     * @param value the int to convert.
     * @return the string conversion.
     */
    public static String integerToString(final int value) {
        return integerToString(value, INTEGER_MAX_PADDING);
    }

    /**
     * Convert the int to a string.
     * @param value the int to convert.
     * @param padSize the pad size.
     * @return the string conversion.
     */
    public static String integerToString(final int value, final int padSize) {
        return (value >= 0 ? POSITIVE_PREFIX : NEGATIVE_PREFIX)
                + StringUtils.leftPad(Integer.toString(value), padSize, PAD_CHAR);
    }

}
