/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugc.api;

/**
 * ValueConstraint is a property constraint which restricts the value of a property based on a {@link ComparisonType
 * ComparisonType} and a value.
 * @param <T> the type of value to be constrained.
 */
public class ValueConstraint<T> extends AbstractPropertyConstraint implements Constraint {

    /**
     * Value to compare against the value of the property.
     */
    private T value;

    /**
     * Type of comparison to perform.
     */
    private ComparisonType comparison;

    /**
     * Creates a new instance of ValueConstraint with the given property name and value. This will create a
     * ValueConstraint which will perform an equals comparison.
     * @param name Name of the property being constrained.
     * @param value Value to compare against the property value.
     */
    public ValueConstraint(final String name, final T value) {
        this(name, value, ComparisonType.Equals);
    }

    /**
     * Creates a new instance of ValueConstraint with the given property name, value and comparison type.
     * @param name Name of the property being constrained.
     * @param value Value to compare against the property value.
     * @param comparison ComparisonType defining the kind of comparison to perform when evaluating property values.
     */
    public ValueConstraint(final String name, final T value, final ComparisonType comparison) {
        this(name, value, comparison, Operator.And);
    }

    /**
     * Creates a new instance of ValueConstraint with the given property name, value and comparison type.
     * @param name Name of the property being constrained.
     * @param value Value to compare against the property value.
     * @param comparison ComparisonType defining the kind of comparison to perform when evaluating property values.
     * @param operator Operator defining how this constraint is joined with other constraints.
     */
    public ValueConstraint(final String name, final T value, final ComparisonType comparison, final Operator operator) {
        super(name, operator);
        this.value = value;
        this.comparison = comparison;
    }

    /**
     * Gets the kind of comparison to perform when evaluating property values.
     * @return ComparisonType which defines how property values should be evaluated.
     */
    public ComparisonType getComparison() {
        return this.comparison;
    }

    /**
     * Gets the value to use when evaluating property values.
     * @return Value to use when evaluating property values.
     */
    public T getValue() {
        return this.value;
    }

    /**
     * Sets the value to use when evaluating property values.
     * @param value Value to use when evaluating property values.
     */
    public void setValue(final T value) {
        this.value = value;
    }

    /**
     * Visit this range constraint.
     * @param constraintVisitor the ConstraintVisitor.
     */
    @Override
    public void accept(final ConstraintVisitor constraintVisitor) {
        constraintVisitor.visitValueConstraint(this);
    }

}
