/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.social.ugcbase.core;

import javax.jcr.InvalidItemStateException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.management.ServiceNotFoundException;

import org.apache.commons.lang3.StringUtils;
import org.apache.sling.api.resource.NonExistingResource;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceWrapper;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.util.tracker.ServiceTracker;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.cq.social.srp.SocialResource;
import com.adobe.cq.social.srp.config.SRPConfigurationError;
import com.adobe.cq.social.srp.config.SRPConfigurationFactory;
import com.adobe.cq.social.srp.config.SocialResourceConfiguration;
import com.adobe.cq.social.ugcbase.SocialUtils;
import com.adobe.granite.security.user.UserProperties;
import com.adobe.granite.security.user.UserPropertiesManager;
import com.day.cq.wcm.webservicesupport.Configuration;

/**
 * Static social utilities.
 */
public final class SocialResourceUtils implements BundleActivator {

    public static String GRAVATAR_PREFIX = "http://www.gravatar.com/avatar/";

    /**
     * Where config definitions are stored. TODO: Temporary This will change once the config location settles down.
     */

    public static final String SRP_CONFIGURATION_ROOT = "/etc/socialconfig/srpc/";
    public static final String SRP_DEFAULT_CONFIG_PATH = SRP_CONFIGURATION_ROOT + "defaultconfiguration";

    private static final Logger LOG = LoggerFactory.getLogger(SocialResourceUtils.class);
    private static ServiceTracker tracker;

    /**
     * Don't ever instantiate this class.
     */
    private SocialResourceUtils() {
    }

    /**
     * Checks a path and an action based on the passed in resolver.
     * @param resolver the resource resolver to use to evaluate the permissions
     * @param path the path to the resource to check
     * @param action the action to check
     * @return True if the resolver is allowed to take the requested action, otherwise false
     */
    public static Boolean checkPermission(final ResourceResolver resolver, final String path, final String action) {
        if (StringUtils.isEmpty(path)) {
            return false;
        }
        try {
            final Session userSession = resolver.adaptTo(Session.class);
            return userSession.hasPermission(path, action);
        } catch (final RepositoryException e) {
            return checkPermission(resolver, StringUtils.substringBeforeLast(path, "/"), action);
        }
    }

    /**
     * Gets the first cloud storage provider. Note: Caller must have read access to configuration storage.
     * @param resolver the resolver to use
     * @return the configuration
     * @deprecated No longer using cloud storage, will return the default configuration.
     */
    @Deprecated
    public static Configuration getFirstCloudStorageConfig(final ResourceResolver resolver) {
        Configuration config = null;
        // Stack trace to locate the origin.
        if (LOG.isDebugEnabled()) {
            LOG.debug("STATIC getDefaultStorageConfig called. Please use SocialUtils", new Throwable(
                "getDefaultStorageConfig"));
        }

        // Note: Caller has to be in admin session
        final Resource configResource = resolver.getResource(SocialResourceUtils.SRP_DEFAULT_CONFIG_PATH);
        SocialResourceConfiguration srConfig = null;
        try {
            if (configResource != null) {
                LOG.debug("config resource is: {}", configResource.getPath());
                try {
                    srConfig = getConfigFactory().createConfiguration(configResource);
                } catch (final ServiceNotFoundException e) {
                    throw new SRPConfigurationError(e);
                }
            }
        } catch (final SRPConfigurationError e) {
            LOG.error("Could not build configuration: ", e);
        }

        if (srConfig != null) {
            config = new ConfigurationImpl(srConfig);
        }
        if (srConfig != null) {
            LOG.debug("getFirstCloudStorageConfig asipath is: {} ", srConfig.getAsiPath());
        }
        return config;

    }

    /**
     * Returns the user properties denoted by the given <code>userId</code>. The user props are looked for using the
     * provided resource resolver, so as to ensure that the user properties are only accessible to users having the
     * necessary access rights on the requested user properties.
     * @param resolver The {@link ResourceResolver}.
     * @param userId The user id for which to retrieve the user properties.
     * @return The {@link UserProperties} or <code>null</code> if not found.
     */
    public static UserProperties getUserProperties(final ResourceResolver resolver, final String userId) {
        UserProperties userProperties = null;
        final UserPropertiesManager upm = resolver.adaptTo(UserPropertiesManager.class);
        if (null != upm && null != userId) {
            try {
                userProperties = upm.getUserProperties(userId, "profile");
            } catch (final RepositoryException e) {
                LOG.warn("User cannot access the profile.", e);
            }
        }
        return userProperties;
    }

    /**
     * Returns the userProperties avatar URL or {@link CollabUtil#DEFAULT_AVATAR} if profile has no avatar defined.
     * @param userProperties The userProperties (may be null)
     * @param absoluteDefaultAvatar The absolute default avatar
     * @param size The avatar size (for example, 32x48)
     * @return the avatar URL or the default one.
     */
    public static String getAvatar(final UserProperties userProperties, final String absoluteDefaultAvatar,
        final SocialUtils.AVATAR_SIZE size) {
        return buildAvatar(userProperties, absoluteDefaultAvatar, size.toString());
    }

    /**
     * Returns the userProperties avatar URL or {@link CollabUtil#DEFAULT_AVATAR} if profile has no avatar defined.
     * @param userProperties The userProperties (may be null)
     * @param absoluteDefaultAvatar The absolute default avatar
     * @param size The avatar size (for example, 32)
     * @return the avatar URL or the default one.
     */
    public static String getAvatar(final UserProperties userProperties, final String absoluteDefaultAvatar,
        final String size) {

        return buildAvatar(userProperties, absoluteDefaultAvatar, size);
    }

    private static String buildAvatar(final UserProperties userProperties, final String absoluteDefaultAvatar,
        final String size) {
        String avatar = SocialUtils.DEFAULT_AVATAR;
        int iSize = SocialUtils.AVATAR_SIZE.THIRTY_TWO.ordinal();
        try {
            iSize = Integer.parseInt(size);
        } catch (final NumberFormatException e) {
            LOG.warn("Bad size passed in, defaulting size.", e);

        }
        if (userProperties != null) {
            try {
                final Resource resource = userProperties.getResource(UserProperties.PHOTOS + "/primary/image");
                if (null != resource) {
                    avatar = resource.getPath() + ".prof.thumbnail." + Integer.toString(iSize) + ".png";
                } else {
                    final String primaryMail = userProperties.getProperty(UserProperties.EMAIL);
                    if (primaryMail != null && !"".equals(primaryMail)) {
                        final String gravatar = getGravatar(primaryMail, absoluteDefaultAvatar);
                        if (gravatar != null) {
                            avatar = gravatar;
                        }
                    }
                }
            } catch (final RepositoryException e) {
                LOG.error("getAvatar: error getting avatar: ", e);
            }
        }
        return avatar;

    }

    /**
     * Returns the gravatar URL.
     * @param email The email address
     * @param absoluteDefaultAvatar The absolute default avatar
     * @return the gravatar URL or the default one.
     */
    private static String getGravatar(final String email, final String absoluteDefaultAvatar) {
        if (StringUtils.isBlank(email) || StringUtils.isBlank(absoluteDefaultAvatar)) {
            return null;
        }
        final StringBuilder sb = new StringBuilder();
        sb.append(GRAVATAR_PREFIX);
        sb.append(email);
        sb.append("?d=").append("mm");
        sb.append("&s=32");
        sb.append("&r=g");
        return sb.toString();
    }

    /**
     * Check if path is non JCR based. Assume that all alternate storage paths have prefix
     * {@link SocialUtils#ASI_UGC_PREFIX}, which is enforced in the cloudconfig UI.
     * @param path the path to the ugc resource
     * @return true if path is of alternate storage type (non jcr)
     */
    public static boolean isCloudUGC(final String path) {
        return StringUtils.startsWith(path, SocialUtils.ASI_UGC_PREFIX);
    }

    /**
     * Check if a given resource is an instance of SocialResource.
     * @param res the resource to check
     * @return true if resource is a SocialResource
     */
    public static boolean isSocialResource(final Resource res) {
        if (res instanceof SocialResource) {
            return true;
        } else if (res instanceof ResourceWrapper) {
            return isSocialResource(((ResourceWrapper) res).getResource());
        } else {
            return false;
        }
    }

    /**
     * Check if a resource is a wrapped resource, and unwrap until a SocialResource is found.
     * @param res resource to unwrap
     * @return an unwrap resource or null if no SocialResource found
     */
    public static SocialResource getSocialResource(final Resource res) {
        if (res == null) {
            return null;
        }
        if (res instanceof SocialResource) {
            return (SocialResource) res;
        } else if (res instanceof ResourceWrapper) {
            return getSocialResource(((ResourceWrapper) res).getResource());
        } else if (res instanceof NonExistingResource) {
            LOG.warn("Resource {} is a NonExistingResource, returning null", res);
            return null;
        } else {
            LOG.debug("Resource {} is unknown resource type, returning null", res);
            return null;
        }
    }

    /**
     * Check to see if the specified exception caused by a jcr InvalidItemStateException. This exception is thrown if
     * any of the changes to be persisted conflicts with a change already persisted through another session and the
     * implementation is such that this conflict can only be detected at save-time and therefore was not detected
     * earlier, at change-time.
     * @param e The exception
     * @return if caused by InvalidItemStateException
     */
    public static boolean causeByInvalidItemStateException(final Exception e) {
        Throwable cause = e;
        do {
            if (cause instanceof InvalidItemStateException) {
                return true;
            }
            cause = cause.getCause();
        } while (cause != null);
        return false;
    }

    private static SRPConfigurationFactory getConfigFactory() throws ServiceNotFoundException {
        final Object factory = tracker.getService();
        if (factory != null && factory instanceof SRPConfigurationFactory) {
            return (SRPConfigurationFactory) factory;
        } else {
            throw new ServiceNotFoundException();
        }
    }

    @Override
    public void start(final BundleContext context) throws Exception {
        synchronized (SocialResourceUtils.class) {
            if (tracker == null) {
                tracker = new ServiceTracker(context, SRPConfigurationFactory.class.getName(), null);
                tracker.open();
            }
        }
    }

    @Override
    public synchronized void stop(final BundleContext context) throws Exception {
        synchronized (SocialResourceUtils.class) {
            if (tracker != null) {
                tracker.close();
            }
        }
    }
}
