/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.activitystreams;

import org.apache.sling.api.resource.ValueMap;

/**
 * <code>Activity</code>
 *
 * Note that property names and descriptions were reused from http://activitystrea.ms/specs/json/1.0/. Although that
 * document specifies how activities are serialized in JSON the names and concepts can be reused. Additional properties
 * and behaviors are added where the JSON abstraction is not suitable.
 *
 * All properties referring to IDs or URLs may return a path instead of an absolute IRI if the activity is still
 * present in the repository. It is the tasks of the exporting servlet to make them absolute.
 */
public interface Activity extends JsonAdaptable {

    /**
     * Provides a permanent, universally unique identifier for the activity in the form of an absolute IRI [RFC3987].
     * An activity SHOULD contain a single id property. If an activity does not contain an id property,
     * consumers MAY use the value of the url property as a less-reliable, non-unique identifier.
     */
    String getId();

    /**
     * An IRI [RFC3987] identifying a resource providing an HTML representation of the activity.
     * An activity MAY contain a url property.
     */
    String getURL();

    /**
     * Returns the path of this activity if stored in the repository.
     * @return the path or <code>null</code>
     */
    String getPath();

    /**
     * Identifies the action that the activity describes. An activity SHOULD contain a verb property whose value is a
     * String that is non-empty and matches either the "isegment-nz-nc" or the "IRI" production in [RFC3339].
     * Note that the use of a relative reference other than a simple name is not allowed. If the verb is not specified,
     * or if the value is null, the verb is assumed to be "post".
     */
    String getVerb();

    /**
     * Natural-language title or headline for the activity encoded as a single String containing HTML markup.
     * An activity MAY contain a title property.
     */
    String getTitle();

    /**
     * Describes the entity that performed the activity.
     * An activity MUST contain one actor property whose value is a single Object.
     *
     * Note that the author object might by dynamically generated based on the user id of the actual author, if present
     * in the system.
     */
    ActivityObject getActor();

    /**
     * Returns the user id of the actor of this activity or <code>null</code> if this object was not created
     * by a user present in this system.
     */
    String getActorUserId();

    /**
     * Natural-language description of the activity encoded as a single String containing HTML markup.
     * Visual elements such as thumbnail images MAY be included. An activity MAY contain a content property.
     */
    String getContent();

    /**
     * Describes the target of the activity. The precise meaning of the activity's target is dependent on the
     * activities verb, but will often be the object the English preposition "to". For instance, in the activity,
     * "John saved a movie to his wishlist", the target of the activity is "wishlist". The activity target MUST NOT be
     * used to identity an indirect object that is not a target of the activity. An activity MAY contain a target
     * property whose value is a single Object.
     */
    ActivityObject getTarget();

    /**
     * Describes the primary object of the activity. For instance, in the activity, "John saved a movie to his wishlist",
     * the object of the activity is "movie". An activity SHOULD contain an object property whose value is a single Object.
     * If the object property is not contained, the primary object of the activity MAY be implied by context.
     */
    ActivityObject getObject();

    /**
     * The date and time at which the activity was published. An activity MUST contain a published property.
     */
    long getPublished();

    /**
     * Describes the application that published the activity. Note that this is not necessarily the same entity that
     * generated the activity. An activity MAY contain a provider property whose value is a single Object.
     */
    ActivityObject getProvider();

    /**
     * Describes the application that generated the activity. An activity MAY contain a generator property whose value
     * is a single Object.
     */
    ActivityObject getGenerator();

    /**
     * The date and time at which a previously published activity has been modified. An Activity MAY contain an updated property.
     */
    long getUpdated();

    /**
     * Description of a resource providing a visual representation of the object, intended for human consumption.
     * The image SHOULD have an aspect ratio of one (horizontal) to one (vertical) and SHOULD be suitable for
     * presentation at a small size. An activity MAY have an icon property.
     */
    MediaLink getIcon();

    /**
     * Returns a read-only value map of the raw properties underlying this activity.
     */
    ValueMap getProperties();

    /**
     * Returns a copy of this activity that is mutable. If this instance is already mutable <code>this</code> is returned.
     * @return a mutable copy of this activity.
     */
    MutableActivity getMutableActivity();

    /**
     * Returns the stream of this activity or <code>null</code> if not accessible or not applicable.
     * @return the stream or null.
     * @since 0.0.10
     */
    ActivityStream getStream();
}