/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.asset.api;

import java.io.InputStream;
import java.util.Iterator;
import java.util.Map;

import org.apache.sling.api.resource.Resource;

import aQute.bnd.annotation.ProviderType;

/**
 * <code>Asset</code> represents a digital resource.
 * <p>
 * Asset is an extension of a Sling Resource, therefore its adaptable.
 * <p>
 * Asset is created by {@link AssetManager} and can be retrieved either via {@link AssetManager} or by
 * adapting a asset {@link Resource} to an <code>Asset</code>.
 * <pre>
 *     eg:
 *     // to create an asset
 *     AssetManager assetManager = resolver.adaptTo(AssetManager.class);
 *     Asset newAsset = assetManager.createAsset("/path/to/asset/document.pdf");
 *
 *     // to get an existing asset
 *     Asset asset = assetManager.getAsset("/path/to/asset/document.pdf");
 *
 *     // to get asset by adapting
 *     Asset asset = resource.adaptTo(Asset.class);
 * </pre>
 * <p>
 * Asset properties can be retrieved or set by adapting it to a ValueMap.
 * <pre>
 *     eg:
 *     // to get Asset properties value map
 *     ValueMap map = asset.adaptTo(ValueMap.class);
 * </pre>
 * <p>
 * Implementor's Note: Implementations are required to provide an adapter to a writable
 * {@link org.apache.sling.api.resource.ValueMap}.
 *
 * @see AssetManager
 * @see org.apache.sling.api.resource.ValueMap
 */
@ProviderType
public interface Asset extends Resource {

    /**
     * Returns a named {@link Rendition} if available.
     *
     * @param name Name of the Rendition to get
     *
     * @return The Rendition object or null if Rendition with the given name cannot be found
     *
     * @throws AssetException If an error occurs trying to load the Rendition object
     * */
    Rendition getRendition(String name);

    /**
     * Returns an iterator for all {@link Rendition}s of this asset.
     *
     * @return The Iterator of renditions or an empty iterator if no renditions has been created.
     */
    Iterator<? extends Rendition> listRenditions();

    /**
     * Get all related Asset with the given name.
     *
     * @param name Name of the related asset collection
     *
     * @return Iterator of related Assets or an empty iterator if relations cannot be found with the given name.
     * */
    Iterator<? extends Asset> listRelated(String name);
    
    /**
     * Get asset relations for a given relation name.
     *
     * @param name Name of the relation
     *
     * @return Iterator of {@link AssetRelation} or an empty iterator if relations cannot be found with the given name.
     * 
     * @since 1.3
     * */
    Iterator<? extends AssetRelation> listRelations(String name);

    /**
     * Returns the metadata of this Asset.
     *
     * @since 1.8
     * */
    AssetMetadata getAssetMetadata();

    /**
     * Adds a new relation to this Asset with additional properties
     *
     * @param name of the relation to be created
     * @param assetPath Path of the Asset to set a relation for
     * @param map to be used to set additional relation properties.
     * 
     * @returns instance of <code>AssetRelation</code>
     * @throws AssetException if Asset relation cannot be set.
     * 
     * @since 1.3
     * */
    AssetRelation addRelation(String name, String assetPath, Map<String, Object> map);
    
    /**
     * Sets a new relation to this Asset.
     *
     * @deprecated As of 1.3, please use {@link #addRelation(String, String)}
     * @param name of the relation to be created
     * @param assetPath Path of the Asset to set a relation for
     * 
     * @throws AssetException if Asset relation cannot be set.
     * 
     */
    @Deprecated
    void setRelation(String name, String assetPath);
    
    /**
     * adds a new relation to this Asset.
     *
     * @param name of the relation to be created
     * @param assetPath Path of the Asset to set a relation for
     * @returns instance of <code>AssetRelation</code>
     * 
     * @throws AssetException if Asset relation cannot be set.
     * @since 1.3
     * */
    AssetRelation addRelation(String name, String assetPath);
    
    /**
     * This method inserts the related asset at <code>srcAssetPath</code>
     * into the related asset list at the position immediately before the related asset at <code>destAssetPath</code>. 
     * 
     * To insert the related asset into last position, <code>destAssetPath</code> can be null.
     * 
     * @param name Name of the relation to be ordered
     * @param srcAssetPath asset path that needs to be moved in the order
     * @param destAssetPath path of the related asset before which the <code>srcAssetPath</code> will be placed.
     * 
     * @throws IllegalArgumentException if srcAssetPath is null or srcAssetPath is not related to Asset.
     *                                  if srcAssetPath and destAssetPath represents the same Asset.
     * @since 1.3
     */
    void orderRelationBefore(String name, String srcAssetPath, String destAssetPath);

    /**
     * Removes Asset relation.
     *
     * @param name of the relation
     * @param assetPath Path of the Asset to be removed from the relations
     *
     * @throws AssetException if relation cannot be deleted
     * */
    void removeRelation(String name, String assetPath);

    /**
     * Removes all relations for the given name.
     *
     * @param name of the relation
     *
     * @throws AssetException if relation cannot be deleted
     * */
    void removeRelation(String name);

    /**
     * Get unique Asset identifier
     *
     * @return Asset identifier as String or null if Asset has no unique identifier
     * */
    String getIdentifier();

    /**
     * Create or update the {@link Rendition}. If a rendition with the given <code>name</code> already exists,
     * it will be replaced. Actual creation of rendition is delegated to the {@link RenditionHandler}.
     *
     * @param name to be used for this rendition
     * @param is InputStream to create this rendition or null if stream is not required by the RenditionHandler
     * @param map to be used by the rendition handler. If the map contains {@link RenditionHandler#PROPERTY_ID},
     * implementation is required to delegate to the RenditionHandler registered with this id. Otherwise, a default
     * rendition handler must be used. Implementations are free to define the semantics of what constitutes a
     * default rendition handler.
     *
     * @return newly created or updated rendition
     *
     * @throws AssetException if new rendition cannot be set
     * @throws NullPointerException if the given map is null
     */
    Rendition setRendition(String name, InputStream is, Map<String, Object> map);

    /**
     * Remove {@link Rendition} with the given name.
     *
     * @param name of the Rendition
     *
     * @throws AssetException if Rendition cannot be removed
     * */
    void removeRendition(String name);

}
