/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.asset.api;

import java.util.Calendar;
import java.util.Iterator;

import org.apache.sling.api.adapter.Adaptable;

/**
 * <code>AssetVersion</code> represents a particular version of an {@link Asset}.
 * <p>
 * AssetVersion is created by {@link AssetVersionManager} and retrieved via
 * {@link AssetVersionManager#getVersion(String)}.
 * <pre>
 *     eg:
 *     // to create a version
 *     AssetVersionManager versionManager = resolver.adaptTo(AssetVersionManager.class);
 *     AssetVersion version = versionManager.createVersion("/path/to/asset/document.pdf", "Some label");
 *
 *     // to get existing versions
 *     Iterator<AssetVersion> versions = versionManager.listVersions("/path/to/asset/document.pdf");
 *     ...
 * </pre>
 * <p>
 * Implementor's Note: Implementations are required to provide an adapter to a read only
 * {@link org.apache.sling.api.resource.ValueMap}
 *
 * @see AssetVersionManager
 * @see org.apache.sling.api.resource.ValueMap
 */
public interface AssetVersion extends Adaptable {

    /**
     * Returns all labels of this version.
     *
     * @return version labels or an empty array if there are none.
     */
    String[] getLabels();

    /**
     * Return the unique version id.
     *
     * @return version id
     */
    String getId();

    /**
     * Get path of this version
     *
     * @return Version path
     * */
    String getPath();

    /**
     * Get path of the parent asset.
     *
     * @return Asset path
     * */
    String getAssetPath();
    
    /**
     * Get name of this version.
     *
     * @return name
     * */
    String getName();

    /**
     * Get the creation date of this version.
     *
     * @return creation date
     */
    Calendar getCreated();

    /**
     * Returns a named {@link Rendition} if available.
     *
     * @param name Rendition name to search
     *
     * @return rendition or null if not found
     *
     * @throws AssetException If an error occurs trying to load the Rendition object
     * */
    Rendition getRendition(String name);

    /**
     * Returns an iterator for all {@link Rendition}s of this asset version.
     *
     * @return The Iterator of renditions or an empty iterator if no renditions has been created.
     */
    Iterator<? extends Rendition> listRenditions();

    /**
     * Get all related Asset with the given name.
     *
     * @param name of the relation
     *
     * @return Iterator of related Assets or an empty iterator if relations cannot be found with the given name.
     * */
    Iterator<Asset> listRelated(String name);
    
    /**
     * Get asset relations for a given relation name.
     *
     * @param name Name of the relation
     *
     * @return Iterator of {@link AssetRelation} or an empty iterator if relations cannot be found with the given name.
     * */
    Iterator<AssetRelation> listRelations(String name);

}
