/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.auth.oauth;

import aQute.bnd.annotation.ConsumerType;
import org.apache.jackrabbit.api.security.user.User;
import org.apache.sling.api.SlingHttpServletRequest;
import org.scribe.builder.api.Api;
import org.scribe.model.OAuthRequest;
import org.scribe.model.Response;

import java.io.IOException;
import java.util.Map;

/**
 * Interface for OAuth providers.
 */
@ConsumerType
public interface Provider {
    /**
     * Currently only oauth 1a and oauth 2 are supported.
     * @see ProviderType
     * @return type
     */
    public ProviderType getType();

    /**
     * Specifies an instance of scribe {@link Api} to use for this provider.
     * @return Api instance
     */
    public Api getApi();

    /**
     * OAuth provider's user details URL
     * @return url
     */
    public String getDetailsURL();

    /**
     * OAuth provider's user extended details URLs, depending on the specific scope
     * @return url
     */
    public String[] getExtendedDetailsURLs(String scope);

    /**
     * OAuth provider's user extended details URLs, depending on the specific scope and previously fetched data
     * (e.g. {@link #getDetailsURL()}, {@link #getExtendedDetailsURLs(String)}).
     * @param scope allows to specify a list of property names for each scope
     * @param userId the userId
     * @param props contains the data previously fetched.
     * @return the list of urls to fetch extended data from.
     */
    String[] getExtendedDetailsURLs(String scope, String userId, Map<String, Object> props);

    /**
     * Unique ID for this provider, used to match a ProviderConfig with this Provider
     * @return  ID of this provider
     */
    public String getId();

    /**
     * Readable name for this Provider
     * @return name of this Provider
     */
    public String getName();

    /**
     * Map the provider's userid to CRX user id; Note that usernames must be unique so
     * the returned username should always include some prefix specific to this provider
     * (e.g. in case facebook and twitter have a user with the same username)
     * @param userId provider's userId
     * @param props map of all provider's properties for this userId
     * @return CQ user id
     */
    public String mapUserId(String userId, Map<String, Object> props);

    /**
     * Return the node path where the user should be created
     * @param userId
     * @param clientId in use when creating this user
     * @param props map of all provider's properties for this user
     * @return relative path to store this user within /home/users (e.g. "facebook/1234" might be appropriate for facebook user with id=12345678)
     */
    public String getUserFolderPath(String userId, String clientId, Map<String, Object> props);

    /**
     * Map the provider's user properties name to CQ user properties.
     * This method will at least be called to map properties fetched from {@link #getDetailsURL()}.
     * If {@link #getExtendedDetailsURLs(String)} is not null, this
     * method will be called for the map of properties fetched from each url.
     * @param srcUrl
     * @param clientId in use to retreive this set of properties
     * @param existing CQ properties that have already been mapped
     * @param newProperties addition provider properties that need to be mapped
     * @return the result of mapping the new properties, and combining with the existing
     */
    public Map<String, Object> mapProperties(String srcUrl, String clientId, Map<String, Object> existing, Map<String, String> newProperties);

    /**
     * Return the property path where the access token will be stored (if ProviderConfig is has access token storage enabled)
     * @param clientId
     * @return the property path where access token may be stored for a user e.g. profile/someapp-clientid/accesstoken
     */
    public String getAccessTokenPropertyPath(String clientId);

    /**
     * Return the property path where the oauth user id will be stored
     * @param clientId
     * @return
     */
    public String getOAuthIdPropertyPath(String clientId);
    
    /**
     * Use the request to get the User who has (or will have) oauth profile data attached
     * @param request
     * @return the User or null, if no User is associated with the request
     */
    public User getCurrentUser(SlingHttpServletRequest request);

    /**
     * Called after a user is updated (e.g. profile data is mapped and applied to the user);
     * @param user
     */
    public void onUserCreate(User user);

    /**
     * Called after a user is created (i.e. profile data is mapped and applied to user already);
     * @param user
     */
    public void onUserUpdate(User user);

    /**
     * Create an OAuthRequest to request protected data from the OAuth provider system.
     * @param url
     * @return the OAuthRequest
     *
     * @since 2.0
     */
    public OAuthRequest getProtectedDataRequest(String url);

    /**
     * Parse the OAuth Response for protected profile data during profile import
     * @param response
     * @return Map of profile properties
     *
     * @since 2.0
     */
    public Map<String, String> parseProfileDataResponse(Response response) throws IOException;

    /**
     * What is the user data property that contains this OAuth provider's user id? (e.g. "id")
     * @return
     */
    public String getUserIdProperty();
    
    /**
     * OAuth provider validate token URL
     * 
     * @param clientId
     * @param token
     * @return url or null if validate token is not supported
     * 
     * @since 2.1
     */
    public String getValidateTokenUrl(String clientId, String token);
    
    /**
     * Check the validity of a token
     * 
     * @param responseBody
     * @param clientId
     * @return true if the response body contains the validity of the token, the token has been issued for the provided clientId
     * and the token type matches with the one provided
     * 
     * @since 2.1
     */
    public boolean isValidToken(String responseBody, String clientId, String tokenType);
    
    /**
     * Parse the response body and return the userId contained in the response
     * 
     * @param responseBody
     * @return the userId contained in the response or null if is not contained
     * 
     * @since 2.1
     */
    public String getUserIdFromValidateTokenResponseBody(String responseBody);
    
    /**
     * Parse the response body and return the error description contained in the response
     * 
     * @param responseBody
     * @return the error description contained in the response or null if is not contained
     * 
     * @since 2.1
     */
    public String getErrorDescriptionFromValidateTokenResponseBody(String responseBody);
    
}
