/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.auth.saml.configuration;

import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;

/**
 * Defines configuration information for the service provider.
 *
 * @author jasonhightower
 */
public class SpConfiguration {

    /**
     * String id that identifies the service provider with the identity provider.
     */
    private String entityId;

    /**
     * Use encryption for SAML Messages
     */
    private boolean useEncryption;

    /**
     * The alias of the SP's private key in the key-store of the 'authentication-service' system user.
     */
    private String spPrivateKeyAlias;

    /**
     * Key-store password of the 'authentication-service' system user
     */
    private char[] keyStorePassword;

    /**
     * Clock tolerance in seconds
     */
    private int clockTolerance;

    /**
     * Creates a new instance of SpConfiguration with the given KeyProvider.
     */
    public SpConfiguration() {
        super();
    }

    /**
     * Sets the id that identifies the service provider with the identity provider.
     *
     * @param entityId String containing the service provider id.
     */
    public void setEntityId(final String entityId) {
        this.entityId = entityId;
    }

    /**
     * Gets the id that identifies the service provider with the identity provider.
     *
     * @return String containing the service provider id.
     */
    public String getEntityId() {
        return this.entityId;
    }

    /**
     * Gets the private Key used to decrypt saml messages.
     *
     * @param keyStore KeyStore bound to 'authentication-service' user to retrieve SP's private key
     * @return Key used to decrypt saml messages.
     */
    public Key getDecryptionKey(KeyStore keyStore) {
        if (keyStore != null) {
            try {
                return keyStore.getKey(spPrivateKeyAlias, keyStorePassword);
            } catch (KeyStoreException e) {
                throw new RuntimeException("Could not retrieve SP's private key from KeyStore.", e);
            } catch (NoSuchAlgorithmException e) {
                throw new RuntimeException("Could not retrieve SP's private key from KeyStore.", e);
            } catch (UnrecoverableKeyException e) {
                throw new RuntimeException("Could not retrieve SP's private key from KeyStore.", e);
            }
        } else {
            throw new RuntimeException("Could not access KeyStore to receive SP's private key.");
        }
    }

    public boolean getUseEncryption() {
        return useEncryption;
    }

    public void setUseEncryption(boolean useEncryption) {
        this.useEncryption = useEncryption;
    }

    public void setSpPrivateKeyAlias(final String spPrivateKeyAlias) {
        this.spPrivateKeyAlias = spPrivateKeyAlias;
    }

    public void setKeyStorePassword(final String keyStorePassword) {
        this.keyStorePassword = keyStorePassword.toCharArray();
    }

    public int getClockTolerance() { return clockTolerance; }
    public void setClockTolerance(int clockTolerance) {
        this.clockTolerance = clockTolerance;
    }
}
