/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.auth.saml.model;

import com.adobe.granite.auth.saml.configuration.SpConfiguration;

import java.util.Calendar;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Assertion {
    private String version;
    private Calendar issueInstant;
    private String id;
    private Issuer issuer;
    private Subject subject;
    private Calendar notBefore;
    private Calendar notOnOrAfter;
    private LinkedList<String> audienceRestrictions;
    private boolean signatureValid;
    private LinkedList<AuthnStatement> authnStatements;

    /**
     * default log
     */
    private final Logger log = LoggerFactory.getLogger(getClass());

    private Map<String, Attribute> attributeStatements;

    public void addAttribute(Attribute attribute) {
        if (null == attributeStatements) {
            this.attributeStatements = new HashMap<String, Attribute>();
        }
        this.attributeStatements.put(attribute.getName(), attribute);
    }

    public Map<String, Attribute> getAttributes() {
        if (null == this.attributeStatements) {
            return Collections.<String, Attribute>emptyMap();
        }
        return this.attributeStatements;
    }

    /**
     * Check if the following properties of the assertion are valid
     * - target audience
     * - time conditions
     * - signature
     *
     * @param spConfiguration The configuration of this service provider
     * @return true if this assertion is valid for this SP
     */
    public boolean isValid(SpConfiguration spConfiguration) {
        // check time conditions
        Calendar now = Calendar.getInstance();

        // add a little time tolerance to compensate for synchronization issues
        now.add(Calendar.SECOND, spConfiguration.getClockTolerance());

        if (notBefore != null && now.before(notBefore)) {
            log.debug("Invalid Assertion: notBefore violated (" + now.toString() + " < " + notBefore.toString() + ").");
            return false;
        }
        if (notOnOrAfter != null && (now.after(notOnOrAfter) || now.equals(notOnOrAfter))) {
            log.debug("Invalid Assertion: notOnOrAfter violated: (" + now.toString() + " >= " + notOnOrAfter.toString() + ").");
            return false;
        }

        // check if the SP entity ID is part of the audience
        if (!audienceRestrictions.contains(spConfiguration.getEntityId())) {
            log.debug("Invalid Assertion: audienceRestrictions violated.");
            return false;
        }

        // check signature
        if (!signatureValid) {
            log.debug("Invalid Assertion: Signature invalid.");
            return false;
        }

        return true;
    }

    public Calendar getNotBefore() {
        return notBefore;
    }

    public void setNotBefore(Calendar notBefore) {
        this.notBefore = notBefore;
    }

    public Calendar getNotOnOrAfter() {
        return notOnOrAfter;
    }

    public void setNotOnOrAfter(Calendar notOnOrAfter) {
        this.notOnOrAfter = notOnOrAfter;
    }

    public void addAudienceRestriction(String audience) {
        if (null == audienceRestrictions) {
            audienceRestrictions = new LinkedList<String>();
        }
        this.audienceRestrictions.add(audience);
    }

    public boolean isSignatureValid() {
        return signatureValid;
    }

    public void setSignatureValid(boolean signatureValid) {
        this.signatureValid = signatureValid;
    }

    public Subject getSubject() {
        return this.subject;
    }

    public void setSubject(Subject subject) {
        this.subject = subject;
    }

    public LinkedList<AuthnStatement> getAuthnStatements() { return this.authnStatements; }

    public void addAuthnStatement(AuthnStatement authnStatement) {
        if (this.authnStatements == null) {
            this.authnStatements = new LinkedList<AuthnStatement>();
        }
        this.authnStatements.add(authnStatement);
    }
}
