/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.auth.saml.util.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Siphon is a simple utility class which reads all available data from an
 * InputStream and writes it to an OutputStream. Siphon is thread safe.
 * 
 * @author jasonhightower
 */
public class Siphon {
	
	/**
	 * Size of the buffer to use when reading/writing data.
	 */
	private int bufferSize;
	
	/**
	 * True if the streams should be closed when siphoning is complete.
	 */
	private boolean closeStreams;
	
	/**
	 * Create a new instance of Siphon with a buffer size of 8192 bytes and
	 * closes the streams after siphoning.
	 */
	public Siphon() {
		this(true);
	}
	
	/**
	 * Create a new instance of Siphon with a buffer size of 8192 bytes.
	 * @param closeStreams True if the streams should be closed after siphoning, false otherwise.
	 */
	public Siphon(final boolean closeStreams) {
		this(closeStreams, 8192);
	}
	
	/**
	 * Creates a new instance of Siphon with the provided buffer size.
	 * @param closeStreams True if the streams should be closed after siphoning, false otherwise.
	 * @param bufferSize Number of bytes for the buffer.
	 */
	public Siphon(final boolean closeStreams, final int bufferSize) {
		super();
		if (bufferSize < 1) {
			throw new IllegalArgumentException("Buffer size must be greater than 0");
		}
		this.bufferSize = bufferSize;
	}
	
	/**
	 * Siphons data from the input stream to the output stream.
	 * @param in InputStream to read data from.
	 * @param out OutputStream to write data to.
	 * @throws IOException An error occurred either reading or writing data.
	 */
	public void siphon(final InputStream in, final OutputStream out) throws IOException {
		if (null == in) {
			throw new IllegalArgumentException("InputStream is required.");
		}
		if (null == out) {
			throw new IllegalArgumentException("OutputStream is required.");
		}
		try {
			final byte[] buffer = new byte[this.bufferSize];
			int read;
			while (-1 != (read = in.read(buffer))) {
				out.write(buffer, 0, read);
			}
		} finally {
			if (this.closeStreams) {
				in.close();
				out.close();
			}
		}
	}
	
}
