/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import aQute.bnd.annotation.ConsumerType;

/**
 * The <code>AbstractCommentCollection</code> implements a {@link CommentCollection} and provides basic implementation
 * of the API, providing abstract methods so that concrete implementations can control storage and retrieval of
 * resources.
 */
@ConsumerType
public abstract class AbstractCommentCollection<C extends Comment> implements CommentCollection<C> {

    private List<C> comments = null;
    private final Resource target;
    private final Resource collection;

    private final AbstractCommentingProvider provider;

    protected AbstractCommentCollection(final Resource target,
                                        final Resource collection,
                                        final AbstractCommentingProvider provider) {
        this.target = target;
        this.collection = collection;
        this.provider = provider;
    }

    /**
     * {@inheritDoc}
     */
    public final C addComment(final String message, final String author, final String annotationData)
            throws CommentException {
        if (StringUtils.isBlank(message)) {
            throw new IllegalArgumentException("message may not be blank");
        }
        return createComment(provider.createCommentResource(getResource(), message, author, annotationData));
    }

    /**
     * {@inheritDoc}
     */
    public final List<C> getCommentList() {
        if (null == comments) {
            comments = new ArrayList<C>();
            final Iterator<Resource> resources = provider.getCommentResources(getResource());
            while (resources.hasNext()) {
                final C comment = createComment(resources.next());
                comments.add(comment);

            }
        }
        return Collections.unmodifiableList(comments);
    }

    public final Calendar getCreated() {
        return getResource().adaptTo(ValueMap.class).get(JcrConstants.JCR_CREATED, Calendar.class);
    }

    public final Calendar getLastModified() {
        return getResource().adaptTo(ValueMap.class).get(JcrConstants.JCR_LASTMODIFIED, Calendar.class);
    }

    /**
     * {@inheritDoc}
     */
    public final String getPath() {
        return getResource().getPath();
    }

    /**
     * {@inheritDoc}
     */
    public final Resource getTarget() {
        return target;
    }

    /**
     * {@inheritDoc}
     */
    public final void remove() {
        provider.removeCollectionResource(getResource());
    }

    /**
     * {@inheritDoc}
     */
    protected final void removeComment(final AbstractComment comment) throws CommentException {
        provider.removeCommentResource(comment.getResource());
        comments = null;
    }

    /**
     * Return the {@link Resource} representing this collection.
     *
     * @return The resource.
     */
    protected final Resource getResource() {
        return collection;
    }

    /**
     * Return the {@link AbstractCommentingProvider} on which this collection is based.
     *
     * @return The provider.
     */
    protected final AbstractCommentingProvider getProvider() {
        return provider;
    }

    /**
     * Returns the implementation object of a {@link Comment} based on the given comment resource.
     *
     * @param commentResource The resource representing the comment.
     *
     * @return The {@link Comment}.
     */
    protected abstract C createComment(final Resource commentResource);
}
