/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import java.util.Calendar;
import java.util.List;

import org.apache.sling.api.resource.Resource;

import aQute.bnd.annotation.ConsumerType;

/**
 * A <code>CommentCollection</code> represents a collection of {@link Comment}s. Comments can be added to and removed
 * from the collection. A collection is tied to a specific target (e.g. the page, asset or node that was commented on) -
 * see {@link #getTarget()}.
 */
@ConsumerType
public interface CommentCollection<C extends Comment> {

    /**
     * Create a new comment and add it to the collection. The comment will be created with the given message, optional
     * author user ID string and optional annotation data.
     *
     * @param message        The message of the comment.
     * @param author         The optional user ID to set as the author of the new comment. If none is provided, the
     *                       session user is considered the author.
     * @param annotationData The optional annotation data. Annotation data may be used to store SVG data when commenting
     *                       on an image that is annotated with such data.
     *
     * @return The comment just created.
     *
     * @throws CommentException Upon encountering an error during comment creation.
     */
    C addComment(String message, String author, String annotationData);

    /**
     * Returns all comments of this collection.
     *
     * @return A {@link List} of comments of this collection, or an empty list if no comments are present.
     */
    List<C> getCommentList();

    /**
     * Returns the moment in time this collection was created.
     *
     * @return The calendar representing the creation date/time.
     */
    Calendar getCreated();

    /**
     * Returns the moment in time this collection was last modified.
     *
     * @return The calendar representing the last modification.
     */
    Calendar getLastModified();

    /**
     * Returns the path of the resource representing this collection.
     *
     * @return A <code>String</code> representing the path.
     */
    String getPath();

    /**
     * Returns the {@link Resource} representing the target this comment collection belongs to.
     *
     * @return The resource.
     */
    Resource getTarget();

    /**
     * Removes (deletes) this comment collection. After calling this method, the collection must be considered
     * invalid/stale and must not be used anymore.
     *
     * @throws CommentException Upon encountering an error while removing.
     */
    void remove();
}
