/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import javax.jcr.RepositoryException;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Properties;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.ReferenceCardinality;
import org.apache.felix.scr.annotations.ReferencePolicy;
import org.apache.felix.scr.annotations.Service;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.servlets.post.PostOperation;
import org.apache.sling.servlets.post.PostResponse;
import org.apache.sling.servlets.post.SlingPostProcessor;
import org.osgi.service.event.EventAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component
@Service
@Properties({
        @Property(name = PostOperation.PROP_OPERATION_NAME, value = "granite:comment"),
        @Property(name = "sling.servlet.methods", value = {"POST"}),
        @Property(name = "sling.servlet.extensions", value = {"html", "json"})
})
public class CommentingOperationHandler implements PostOperation {
    /**
     *
     */
    private static final long serialVersionUID = -5076236106916461224L;

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(CommentingOperationHandler.class);

    @Reference
    private CommentManager cm = null;

    @Reference(cardinality=ReferenceCardinality.OPTIONAL_UNARY, policy=ReferencePolicy.DYNAMIC)
    private volatile EventAdmin eventAdmin = null;

    private final String DEFAULT_CREATOR = "anonymous";
    private final String MESSAGE = "message";
    private final String ANNOTATION_DATA = "annotationData";

    public void run(SlingHttpServletRequest request, PostResponse response, SlingPostProcessor[] processors) {
        Resource resource = request.getResource();
        String createdBy;

        try {
            createdBy = getAuthor(request);
        } catch (RepositoryException re) {
            log.error("Unable to find ID of author", re);
            response.setError(re);
            return;
        }

        String path = resource.getPath();
        String message = request.getParameter(MESSAGE);
        String annotationData = request.getParameter(ANNOTATION_DATA);

        // create the comment
        try {
            Comment comment = createComment(resource, path, message, annotationData, createdBy);
            EventAdmin ea = this.eventAdmin;
            if (ea != null && comment!=null) {
                ea.postEvent(CommentingEvent.commented(comment.getPath()).toEvent());
            } else if(ea == null){
                log.debug("event admin is null");
            }else{
                log.debug("comment could not be created");
            }
        } catch (Exception e) {
            log.error("Unable to create comment.", e);
            response.setError(e);
        }
    }

    private String getAuthor(SlingHttpServletRequest request) throws RepositoryException {
        Authorizable auth = request.getResourceResolver().adaptTo(Authorizable.class);
        String createdBy = DEFAULT_CREATOR;
        if (auth != null) {
            createdBy = auth.getID();
        }
        return createdBy;
    }

    // Create comment using Granite commenting
    private Comment createComment(Resource resource, String path, String message, String annotationData, String createdBy)
            throws Exception {
        if (path == null || message == null) {
            throw new IllegalArgumentException("Path and message may not be null");
        }

        return cm.getOrCreateCollection(resource, CommentCollection.class).addComment(message, createdBy, annotationData);
    }

    void bindEventAdmin(final EventAdmin ea) {
        eventAdmin = ea;
        log.debug("binding Event Admin");
    }

    void unbindEventAdmin(final EventAdmin ea) {
        if ( this.eventAdmin == ea ) {
            eventAdmin = null;
            log.debug("un-binding Event Admin");
        }
    }
}
