/*
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.granite.contexthub.api;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;
import java.util.Set;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import com.adobe.granite.contexthub.impl.ClientLib;
import com.adobe.granite.contexthub.impl.Constants;

/**
 * This class is a bean providing information of the ContextHub's store.
 */
public class Store {
    /* defines if store is enabled or disabled */
    private boolean enabled;

    /* store name */
    private String name = "";

    /* path (for example: /etc/cloudsettings/default/contexthub/store-name) */
    private String path;

    /* resource type (for example: granite/contexthub/components/stores/store-name) */
    private String resourceType;

    /* map of properties */
    private ValueMap properties;

    /* list of run modes in which given store should be initialized */
    private Set<String> runModes;

    /**
     * Constructs bean with information of specified store.
     *
     * @param resource store resource
     */
    public Store(Resource resource) {
        if (resource != null) {
            this.properties = ResourceUtil.getValueMap(resource);

            /* get store properties */
            ValueMap properties = getProperties();
            String[] runModesProperty = properties.get(Constants.STORE_RUN_MODES, String[].class);

            this.enabled = properties.get(Constants.STORE_ENABLED, false);
            this.name = properties.get(Constants.STORE_TYPE, String.class);
            this.path = resource.getPath();
            this.resourceType = resource.getResourceType();
            this.runModes = new TreeSet<String>();

            if (runModesProperty != null) {
                this.runModes.addAll(Arrays.asList(runModesProperty));
            }
        }
    }

    /**
     * Returns true if store is enabled.
     *
     * @return {@code true} if store is enabled
     */
    public boolean isEnabled() {
        return enabled;
    }

    /**
     * Returns store name.
     *
     * @return store name
     */
    public String getName() {
        return name;
    }

    /**
     * If this store was created from a resource, returns store path (for example:
     * /etc/cloudsettings/default/contexthub/store-name). Otherwise, returns {@code null}.
     *
     * @return store path, or {@code null}
     */
    public String getPath() {
        return path;
    }

    /**
     * Returns list of run modes in which store should be available.
     *
     * @return list of run modes
     */
    public Set<String> getRunModes() {
        return runModes;
    }

    /**
     * Returns store resource type (for example: granite/contexthub/components/stores/store-name).
     *
     * @return store resource type
     */
    public String getResourceType() {
        return resourceType;
    }

    /**
     * Returns store properties.
     *
     * @return {@code ValueMap} containing store properties
     */
    public ValueMap getProperties() {
        return properties;
    }

    /**
     * Set whether this store should be enabled or not.
     *
     * @param enabled whether this store should be enabled or not
     */
    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    /**
     * Set the store's name.
     *
     * @param name the store's name
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Set all &amp; any properties for this store.
     *
     * @param properties the properties
     */
    public void setProperties(ValueMap properties) {
        this.properties = properties;
    }

    /**
     * Returns store specific kernel client libraries.
     *
     * @return {@code List} of kernel client libraries used by the store
     */
    public List<String> getCategories() {
        List<String> clientLibs = new ArrayList<String>();

        clientLibs.add(ClientLib.buildClientlibName(Constants.CONTEXTHUB_CLIENTLIB_STORE, getName()));
        clientLibs.add(ClientLib.buildClientlibName(Constants.CONTEXTHUB_CLIENTLIB_STORE, getName(), Constants.CONTEXTHUB_CLIENTLIB_OVERRIDE_POSTFIX));

        return clientLibs;
    }

    /**
     * Overrides default {@code toString()} and returns store specific information.
     *
     * @return store's {@code name}, {@code path} and {@code resourceType}
     */
    public String toString() {
        return String.format("'%s' (path=%s, resourceType=%s)", getName(), getPath(), getResourceType());
    }
}
