/*******************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ******************************************************************************/
package com.adobe.granite.offloading.api;

import org.apache.sling.discovery.InstanceDescription;

import java.util.Map;
import java.util.Set;

/**
 * The {@code OffloadingTopicManager} manages instance blacklisting / whitelisting for specific {@code topics}. Furthermore, it provides a
 * view of the instances grouped by topic.
 *
 * Clients outside offloading must not use this interface.
 */
public interface OffloadingTopicManager {

    /**
     * Provides a {@link Map} view of all the instances from the {@code topology}.
     *
     * @return a map of instances grouped by topic
     */
    Map<String, TopicInstancesHolder> getInstances();

    /**
     * <p>
     * Configures the topics an instance will process using a configuration map structured in the following way:
     * </p>
     * <ul>
     *     <li>
     *         {@code key} - one of: {@code exclusive}, {@code enable}, {@code disable} (for more details see
     *         {@link TopicConfigurationAction}
     *     </li>
     *     <li>
     *         {@code value} - a set of topic strings
     *     </li>
     * </ul>
     * <p>
     * The topics marked as {@code exclusive} will be added to the exclusive whitelist. The topics marked otherwise will just be added /
     * removed from the blacklist.
     * </p>
     *
     * @param slingID       the {@code slingID} of the instance to be configured
     * @param configuration a configuration map.
     * @throws OffloadingException
     *          if the configuration operation fails
     */
    void configureInstance(String slingID, Map<TopicConfigurationAction, Set<String>> configuration);

    /**
     * Retrieves the topics registered for processing from an instance.
     *
     * @param instance the instance
     * @return a set of topics
     */
    Set<String> getRegisteredTopics(InstanceDescription instance);

    /**
     * Retrieves the whitelisted topics from an instance.
     *
     * @param instance the instance
     * @return a set of topics
     */
    Set<String> getWhitelistedTopics(InstanceDescription instance);

    /**
     * Retrieves the blacklisted topics from an instance.
     *
     * @param instance the instance
     * @return a set of topics
     */
    Set<String> getBlacklistedTopics(InstanceDescription instance);
}
