/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *************************************************************************/
package com.adobe.granite.offloading.workflow.api;

import com.adobe.granite.workflow.metadata.MetaDataMap;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * Helper class for integrating generic job execution and workflow offloading into workflow steps.
 */
public final class WorkflowOffloadingHelper {

    private static final String OFFLOADING_PAYLOAD_DELIMITER = ",";

    /**
     * <p>Read the job topic from the given process arguments.</p>
     * <p>Expects the job topic be set using {@link JobOffloadingProcessArguments#JOB_TOPIC}.</p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return The job topic, or <code>null</code> if the job topic is not set.
     */
    public static String getJobTopic(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(JobOffloadingProcessArguments.JOB_TOPIC.getArgumentName(), String.class);
        }
        return null;
    }

    /**
     * <p>Read the job properties from the given process arguments.</p>
     * <p>Expects the job properties be set using {@link JobOffloadingProcessArguments#JOB_PROPERTIES}.</p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return ValueMap containing the job properties from the process arguments. If the job properties are not set, an
     *         empty ValueMap is returned.
     */
    public static ValueMap getJobProperties(MetaDataMap arguments) {
        ValueMap jobProperties = new ValueMapDecorator(new HashMap<String, Object>());
        if (arguments != null) {
            String[] properties = arguments.get(JobOffloadingProcessArguments.JOB_PROPERTIES.getArgumentName(), new String[0]);
            for (String property : properties) {
                String[] split = property.split("\\|");
                if (split.length == 2) {
                    jobProperties.put(split[0], split[1]);
                }
            }
        }
        return jobProperties;
    }

    /**
     * <p>Read the generic offloading input payload from the given process arguments.</p>
     * <p>Expects the offloading input payload be set using {@link JobOffloadingProcessArguments#OFFLOADING_INPUT}.</p>
     *
     * @param arguments The workflow process arguments. Must not be <code>null</code>.
     *
     * @return String array containing the offloading input payload, or an empty array, if the offloading input payload
     *         is not set.
     */
    public static String[] getOffloadingInput(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(JobOffloadingProcessArguments.OFFLOADING_INPUT.getArgumentName(), new String[0]);
        }
        return new String[0];
    }

    /**
     * <p>Read the generic offloading output payload from the given process arguments.</p>
     * <p>
     * Expects the offloading output payload be set using {@link JobOffloadingProcessArguments#OFFLOADING_OUTPUT}.
     * </p>
     *
     * @param arguments The workflow process arguments. Must not be <code>null</code>.
     *
     * @return String array containing the offloading output payload, or an empty array, if the offloading output
     *         payload is not set.
     */
    public static String[] getOffloadingOutput(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(JobOffloadingProcessArguments.OFFLOADING_OUTPUT.getArgumentName(), new String[0]);
        }
        return new String[0];
    }

    /**
     * Converts the given payload array into a flat string for use on the job properties.
     *
     * @param offloadingPayload The offloading payload array.
     *
     * @return Flat string representation of the given array. If the given payload array is <code>null</code> or empty,
     *         an empty String is returned.
     */
    public static String convertToOffloadingPayload(String[] offloadingPayload) {
        StringBuffer inputBuffer = new StringBuffer();
        if (offloadingPayload != null && offloadingPayload.length > 0) {
            for (String path : offloadingPayload) {
                inputBuffer.append(path).append(OFFLOADING_PAYLOAD_DELIMITER);
            }
        }
        return inputBuffer.toString();
    }

    /**
     * <p>Read the workflow model from the given process arguments.</p>
     * <p>
     * Expects the workflow model be set using {@link WorkflowOffloadingProcessArguments#WORKFLOW_MODEL}.
     * </p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return The workflow model or <code>null</code> if the workflow model is not set.
     */
    public static String getWorkflowModel(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(WorkflowOffloadingProcessArguments.WORKFLOW_MODEL.getArgumentName(), String.class);
        }
        return null;
    }

    /**
     * <p>Read the workflow payload from the given process argument or work item.</p>
     * <p>
     * Expects the workflow payload be set using {@link WorkflowOffloadingProcessArguments#WORKFLOW_PAYLOAD}. If not set
     * on the arguments, it falls back to use the fallback payload.
     * </p>
     *
     * @param fallbackPayload The workflow workflow payload to use as a fallback.
     * @param arguments       The workflow process arguments.
     *
     * @return String The workflow payload or <code>null</code> if the fallback payload is not set.
     */
    public static String getWorkflowPayload(String fallbackPayload, MetaDataMap arguments) {
        String payload = null;
        //first try from arguments
        if (arguments != null) {
            payload = arguments.get(WorkflowOffloadingProcessArguments.WORKFLOW_PAYLOAD.getArgumentName(), String.class);
            //fallback
            if (StringUtils.isBlank(payload)) {
                payload = fallbackPayload;
            }
        }
        return payload;
    }

    /**
     * <p>Whether to include the workflow model to the offloading input.</p>
     * <p>
     * Expects the flag be set using {@link WorkflowOffloadingProcessArguments#OFFLOADING_INPUT_WORKFLOW_MODEL}.
     * </p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return <code>true</code> if the workflow model must be included, <code>false</code> otherwise. Returns
     *         <code>false</code> if the property is not set at all.
     */
    public static boolean doTransportWorkflowModelWithInput(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(WorkflowOffloadingProcessArguments.OFFLOADING_INPUT_WORKFLOW_MODEL.getArgumentName(), Boolean.FALSE);
        }
        return false;
    }

    /**
     * <p>Whether to include the workflow model to the offloading output.</p>
     * <p>
     * Expects the flag be set using {@link WorkflowOffloadingProcessArguments#OFFLOADING_OUTPUT_WORKFLOW_MODEL}.
     * </p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return <code>true</code> if the workflow model must be included, <code>false</code> otherwise. Returns
     *         <code>false</code> if the property is not set at all.
     */
    public static boolean doTransportWorkflowModelWithOutput(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(WorkflowOffloadingProcessArguments.OFFLOADING_OUTPUT_WORKFLOW_MODEL.getArgumentName(), Boolean.FALSE);
        }
        return false;
    }

    /**
     * <p>Whether to include the workflow payload to the offloading input.</p>
     * <p>
     * Expects the flag be set using {@link JobOffloadingProcessArguments#OFFLOADING_INPUT_WORKFLOW_PAYLOAD}.
     * </p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return <code>true</code> if the workflow payload must be included, <code>false</code> otherwise. Returns
     *         <code>false</code> if the property is not set at all.
     */
    public static boolean doTransportWorkflowPayloadWithInput(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(JobOffloadingProcessArguments.OFFLOADING_INPUT_WORKFLOW_PAYLOAD.getArgumentName(), Boolean.FALSE);
        }
        return false;
    }

    /**
     * <p>Whether to include the workflow payload to the offloading output.</p>
     * <p>
     * Expects the flag be set using {@link JobOffloadingProcessArguments#OFFLOADING_OUTPUT_WORKFLOW_PAYLOAD}.
     * </p>
     *
     * @param arguments The workflow process arguments.
     *
     * @return <code>true</code> if the workflow payload must be included, <code>false</code> otherwise. Returns
     *         <code>false</code> if the property is not set at all.
     */
    public static boolean doTransportWorkflowPayloadWithOutput(MetaDataMap arguments) {
        if (arguments != null) {
            return arguments.get(JobOffloadingProcessArguments.OFFLOADING_OUTPUT_WORKFLOW_PAYLOAD.getArgumentName(), Boolean.FALSE);
        }
        return false;
    }

    /**
     * <p>Builds the complete workflow offloading input payload.</p>
     * <p>
     * Puts the generic offloading input along with the given workflow payload and workflow model on the offloading
     * input payload.
     * </p>
     *
     * @param arguments       The workflow process arguments. Must not be <code>null</code>.
     * @param workflowPayload The workflow payload to put on the offloading input.
     * @param workflowModel   The workflow model to put on the offloading input.
     *
     * @return The workflow offloading input. Returns <code>null</code> if the given process arguments is
     *         <code>null</code>.
     */
    public static String getWorkflowOffloadingInput(MetaDataMap arguments, String workflowPayload, String workflowModel) {
        String offloadingInputPayload = null;
        if (arguments != null) {
            String[] fromArguments = WorkflowOffloadingHelper.getOffloadingInput(arguments);
            List<String> offloadingInput = new ArrayList<String>();
            if (WorkflowOffloadingHelper.doTransportWorkflowPayloadWithInput(arguments) && StringUtils.isNotBlank(workflowPayload)) {
                offloadingInput.add(workflowPayload);
            }
            if (WorkflowOffloadingHelper.doTransportWorkflowModelWithInput(arguments) && StringUtils.isNotBlank(workflowModel)) {
                offloadingInput.add(workflowModel);
            }
            String[] offloadingInputArray = offloadingInput.toArray(new String[offloadingInput.size()]);
            String[] all = (String[]) ArrayUtils.addAll(fromArguments, offloadingInputArray);

            offloadingInputPayload = WorkflowOffloadingHelper.convertToOffloadingPayload(all);
        }
        return offloadingInputPayload;
    }

    /**
     * <p>Builds the complete workflow offloading output payload.</p>
     * <p>
     * Puts the generic offloading output along with the given workflow payload and workflow model on the offloading
     * output payload.
     * </p>
     *
     * @param arguments       The workflow process arguments. Must not be <code>null</code>.
     * @param workflowPayload The workflow payload to put on the offloading output.
     * @param workflowModel   The workflow model to put on the offloading output.
     *
     * @return The workflow offloading output. Returns <code>null</code> if the given process arguments is
     *         <code>null</code>.
     */
    public static String getWorkflowOffloadingOutput(MetaDataMap arguments, String workflowPayload, String workflowModel) {
        String offloadingOutputPayload = null;
        if (arguments != null) {
            String[] fromArguments = WorkflowOffloadingHelper.getOffloadingOutput(arguments);

            List<String> offloadingOutput = new ArrayList<String>();
            if (WorkflowOffloadingHelper.doTransportWorkflowPayloadWithOutput(arguments) && StringUtils.isNotBlank(workflowPayload)) {
                offloadingOutput.add(workflowPayload);
            }
            if (WorkflowOffloadingHelper.doTransportWorkflowModelWithOutput(arguments) && StringUtils.isNotBlank(workflowModel)) {
                offloadingOutput.add(workflowModel);
            }
            String[] offloadingOutputArray = offloadingOutput.toArray(new String[offloadingOutput.size()]);
            String[] all = (String[]) ArrayUtils.addAll(fromArguments, offloadingOutputArray);

            offloadingOutputPayload = WorkflowOffloadingHelper.convertToOffloadingPayload(all);
        }
        return offloadingOutputPayload;
    }
}
