/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.omnisearch.api.core;

import aQute.bnd.annotation.ProviderType;
import com.adobe.granite.omnisearch.api.suggestion.SuggestionResult;
import com.day.cq.search.result.SearchResult;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import com.day.cq.i18n.I18n;

import java.util.Iterator;
import java.util.Map;

/**
 * OmniSearch enables different search modules (or location) to plugin with common and unified search interface i.e. OmniSearch.
 * A search module generally handles search related to a one or more resource type i.e. Asset, Site, User, Group, Asset Collection
 * etc.
 *
 * OmniSearch enables couple of key functionalities
 * <ul>
 *     <li>Suggestions &amp; SpellCheck</li>
 *     <li>Predicate Suggestions</li>
 *     <li>Common Search - search across all modules</li>
 *     <li>Search within a particular module i.e. Asset</li>
 *     <li>Saved Search functionality for the module that implements <code>SavedSearchHandler</code></li>
 * </ul>
 * Each module has the flexibility to handle/specify query by registering a <code>OmniSearchHandler</code>
 */
@ProviderType
public interface OmniSearchService {

    /**
     * Location has 1:1 mapping with a search module. <code>location</code> in
     * search request parameter, can specify particular search module to search.
     * i.e. if location = "x", then omnisearch would only search module that has ID = "x"
     * @see com.adobe.granite.omnisearch.spi.core.OmniSearchHandler#getID()
     */
    public static final String LOCATION = "location";

    static final String MIN_LENGTH_SUGGESTION_PROPERTY = "omnisearch.suggestion.requiretext.min";

    static final String SPELLCHECK_REQUIRE_PROPERTY = "omnisearch.suggestion.spellcheck.require";

    static final int MIN_LENGTH_FOR_SUGGESTION = 3;

    static final boolean SPELLCHECK_REQUIRE = false;
    
    /**
     *
     * This function returns the search result for OmniSearch.
     *
     * Depending on <code>predicateParamters</code>, it could contain results from multiple
     * search modules or one of the search module.
     *
     * If user does not have access to any module, that module will
     * not be present in search Results.
     * @param resolver <code>ResourceResolver</code> instance
     * @param predicateParameters Map of parameters this should be in format of &lt;String, String&gt; or  &lt;String, String[]&gt;
     * @param limit number of result on a page
     * @param offset pffset for next page result  @return
     * @return Return Results in format <code>Map</code> with Key as location @see com.adobe.granite.omnisearch.api.core.OmniSearchService#LOCATION ,
     * and Value as <code>SearchResult</code>.
     */
    public Map<String, SearchResult> getSearchResults(ResourceResolver resolver, Map<String, ?> predicateParameters, long limit, long offset);


    /**This function will return suggestions for omnisearch.
     * Suggestion Result will be in format of <code>Map</code> with key as String
     * and Value as <code>SuggestionResult</code>
     * If user does not have access to any module, that module will
     * not be present in search Results.
     * @param resolver <code>ResourceResolver</code> instance
     * @param i18n I18n instance
     * @param term text term for which suggestions are require
     * @param location this is unique id for each module @see com.adobe.granite.omnisearch.api.core.OmniSearchService#LOCATION.
     * @return Map
     */
    public Map<String, SuggestionResult> getSuggestions(ResourceResolver resolver, I18n i18n, String term, String location);


    /**
     * This function will return all search modules and their configuration
     * that are registered with OmniSearch.
     * @param resolver <code>ResourceResolver</code> instance
     * @return a map with key as <code>location</code> and value as configuration resource
     * @see com.adobe.granite.omnisearch.spi.core.OmniSearchHandler#getModuleConfig(org.apache.sling.api.resource.ResourceResolver)
     */
    public Map<String, Resource> getModules(ResourceResolver resolver);

    /**
     * This function will return the configuration <code>Resource</code> of the
     * search modules that is registered with OmniSearch.
     * @param resolver <code>ResourceResolver</code> instance
     * @param location this is unique id for each module @see com.adobe.granite.omnisearch.api.core.OmniSearchService#LOCATION.
     * @return resource
     */
    public Resource getModuleConfiguration(ResourceResolver resolver, String location);

    /**This function returns the list of saved searches for the particular module that is register with
     * omnisearch and has <code>SavedSearchHandler</code> implemented
     * @param resolver <code>ResourceResolver</code> instance
     * @param location this is unique id for each module @see com.adobe.granite.omnisearch.api.core.OmniSearchService#LOCATION.
     * @param limit number of result on a page
     * @param offset offset for next page result
     * @return <Code>Iterator</Code> of <code>Resource</code>  of nodes of saved search for the module.
     * @throws OmniSearchException instance of <code>OmniSearchException</code> which provide error from the module
     */
    public Iterator<Resource> getSavedSearches(ResourceResolver resolver, String location, long limit, long offset) throws OmniSearchException;

    /** This function returns the details of saved <code>Query</code> parameters for the particular saved search.
     *
     * @param resolver <code>ResourceResolver</code> instance
     * @param location this is unique id for each module @see com.adobe.granite.omnisearch.api.core.OmniSearchService#LOCATION
     * @param path  path of the node of saved search, from which parameters of saved search <code>Query</code> required
     * @return <code>Map</code> containing parameters of the <code>Query</code>
     * @throws OmniSearchException instance of <code>OmniSearchException</code> which provide error from the module
     */
    public Map<String, String> getSavedSearchParameters(ResourceResolver resolver, String location, String path) throws OmniSearchException;

    /** This function creates a  new saved search or updates a existing saved search based on the
     * parameters provided in predicateMap parameter.
     *
     * @param resolver <code>ResourceResolver</code> instance
     * @param requestParameters <code>Map</code> of the parameters in format &lt;String, String&gt; or  &lt;String, String[]&gt;
     *                            containing all parameters that needed to be saved as <code>Query</code> parameter
     * @return <code>ResourceResolver</code> instance
     * @throws OmniSearchException instance of <code>OmniSearchException</code> which provide error from the module
     */
    public Resource createOrUpdateSavedSearch(ResourceResolver resolver, Map<String, Object> requestParameters) throws OmniSearchException;

    /**
     * This function delete the saved search node existing on the provided path
     * @param resolver <code>ResourceResolver</code> instance
     * @param location this is unique id for each module @see com.adobe.granite.omnisearch.api.core.OmniSearchService#LOCATION
     * @param path  path of the node of saved search to delete
     * @return true, if node deleted successfully, false otherwise.
     * @throws OmniSearchException instance of <code>OmniSearchException</code> which provide error from the module
     */
    public boolean deleteSavedSearch(ResourceResolver resolver, String location, String path) throws OmniSearchException;

}
