/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.security.user;

import org.apache.sling.api.resource.Resource;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import java.util.Iterator;

import aQute.bnd.annotation.ProviderType;

/**
 * <p>
 * UserProperties is a general interface used to represent user preferences
 * and profiles. It provides convenience methods to retrieve properties
 * in various formats.
 * </p>
 * <p>
 * In addition this interface allows to access the underlying JCR Node used
 * to write or modify this <code>UserProperties</code>, version or lock it
 * as well as to register observation event listeners.
 * </p>
 */
@ProviderType
public interface UserProperties {

    /**
     * Constant for the property containing a description of an authorizable.
     */
    String ABOUT_ME = "aboutMe";

    /**
     * Constant for the email property of an authorizable.
     */
    String EMAIL = "email";

    /**
     * Constant for the family name of a given user.
     */
    String FAMILY_NAME = "familyName";

    /**
     * Constant for the given name of a given user or group.
     */
    String GIVEN_NAME = "givenName";

    /**
     * Constant for the middle name property of a user.
     */
    String MIDDLE_NAME = "middleName";

    /**
     * Constant for the display name (or nick name) property of a given user
     * or group. See also {@link #getDisplayName()} for a implementation
     * specific calculation of a display name that may rely on or fallback to
     * various name related properties.
     */
    String DISPLAY_NAME = "displayName";

    /**
     * Constant for the user property storing images of a user.
     */
    String PHOTOS = "photos";


    /**
     * Returns the ID of the authorizable this user properties belong to.
     *
     * @return the ID of the authorizable this user properties belong to.
     */
    String getAuthorizableID();

    /**
     * Returns the authorizables path
     *
     * @return the path of the authorizable.
     */
    String getAuthorizablePath();

    /**
     * Returns the JCR <code>node</code> associated with this user properties
     * instance. The node may then be used to write properties or execute other
     * JCR specific operations on this user properties that are not covered by
     * convenience methods defined by this interface.
     *
     * @return the <code>node</code> associated with this user properties instance.
     */
    Node getNode();

    /**
     * Retrieves the property names directly available with this instance.
     * This is the same as {@link #getPropertyNames(String)} where the relative
     * path is <code>null</code>.
     *
     * @return An array of property names.
     * @throws RepositoryException If an error occurs.
     */
    String[] getPropertyNames() throws RepositoryException;

    /**
     * Retrieves the property names available at the specified relative path.
     *
     * @param relPath A relative path to the sub tree where to retrieve the
     * property names.
     * @return An array of property names.
     * @throws RepositoryException If an error occurs.
     */
    String[] getPropertyNames(String relPath) throws RepositoryException;

    /**
     * <p>
     * Returns the string value of the property identified by the specified
     * <code>relativePath</code>. This may be a simple property name in case
     * of {@link #getPropertyNames()} or the relative path to the property
     * in case of {@link #getPropertyNames(String)}.
     * </p>
     * <p>
     * Note that this method is a simplified variant of
     * {@link #getProperty(String, Object, Class)} where the default value is
     * <code>null</code> and the desired type <code>String</code>.
     * </p>
     *
     * @param relativePath Relative path referring to the property to
     * be retrieved.
     * @return The string value of the specified property or <code>null</code>.
     * @throws RepositoryException If an error occurs.
     */
    String getProperty(String relativePath) throws RepositoryException;

    /**
     * Returns the property at the specified relative path. The expected
     * format of the property value is specified by the type parameter. If no
     * property exists at <code>relativePath</code> the specified
     * <code>defaultValue</code> is returned.
     *
     * @param relativePath Relative path referring to the property to
     * be retrieved.
     * @param defaultValue The default value to be used if the property at
     * relativePath does not exist.
     * @param type The expected value type. The property value may need to
     * be converted.
     * @param <T> specifies the expected format of the property
     * @return The value of the specified property converted to the specified
     * <code>type</code> or the <code>defaultValue</code> if the property does
     * not exist or cannot be converted to the specified type.
     * @throws RepositoryException If an error occurs.
     */
    <T> T getProperty(String relativePath, T defaultValue, Class<T> type) throws RepositoryException;

    /**
     * Retrieve the resource at the specified path relative to this instance.
     *
     * @param relativePath Relative path to the resource to be retrieved.
     * @return The resource at the specified relative path or <code>null</code>
     * if the the path cannot be resolved to a resource.
     * @throws RepositoryException If an error occurs.
     */
    Resource getResource(String relativePath) throws RepositoryException;

    /**
     * Retrieve the resources at the specified path relative to this instance.
     *
     * @param relativePath Relative path to the resources to be retrieved.
     * @return The resources at the specified relative path or <code>null</code>
     * if the the path cannot be resolved to resources.
     * @throws RepositoryException If an error occurs.
     */
    Iterator<Resource> getResources(String relativePath) throws RepositoryException;

    /**
     * Calculates the resource path for the property located the specified
     * <code>relativePath</code> using the given <code>suffix</code>. If the
     * property does not exist the specified <code>defaultPath</code> is returned.
     *
     * @param relativePath Relative path to the property for which the resource
     * path will be calculated.
     * @param suffix An optional suffix to be appended to the resource path.
     * @param defaultPath An optional default path in case the requested
     * property does not exist (or is not accessible).
     * @return The resource path for the requested property or the default path.
     * @throws RepositoryException If an error occurs.
     */
    String getResourcePath(String relativePath, String suffix, String defaultPath) throws RepositoryException;

    /**
     * Calculate a display name from this <code>UserProperties</code> instance.
     *
     * @return Implementation specific display name as stored in this user
     * properties instance or some implementation specific default.
     * @throws RepositoryException If an error occurs.
     */
    String getDisplayName() throws RepositoryException;

    /**
     * Calculate a display name from this <code>UserProperties</code> instance.
     *
     * @param nameDisplayOrder Order of given, middle and family names.
     * Western name order should be "givenName middleName familyName",
     * Eastern name order should be "familyName givenName middleName".
     * @return Implementation specific display name as stored in this user
     * properties instance or some implementation specific default.
     * @throws RepositoryException If an error occurs.
     */
    String getDisplayName(String nameDisplayOrder) throws RepositoryException;

    /**
     * Checks if the authorizable of this properties is a group.
     * @return {@code true} if this properties belongs to a group.
     */
    boolean isGroupProperties();
}
