/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.security.user;

import javax.jcr.RepositoryException;
import java.util.Collection;

import aQute.bnd.annotation.ProviderType;

/**
 * <p>
 * The <code>UserPropertiesComposite</code> represents a composite (aggregation) of {@link UserProperties}. A composite
 * can be retrieved via {@link UserPropertiesManager#getUserPropertiesComposite(String, String[])} or {@link
 * UserPropertiesManager#getUserPropertiesComposite(String, String[])}.
 * </p>
 * <p>
 * The composite transparently accesses properties within the user properties contained in the composite.
 * </p>
 */
@ProviderType
public interface UserPropertiesComposite {

    /**
     * Retrieves all first-level property names found on all user properties contained in the composite.
     *
     * @return An unordered collection of property names, or an empty collection if no property names were found.
     *
     * @throws RepositoryException If an error occurs accessing the user properties.
     */
    Collection<String> getPropertyNames() throws RepositoryException;

    /**
     * Retrieves all property names found at the given <code>relativePath</code> within all user properties contained
     * within this composite.
     *
     * @param relPath The relative path of a user properties node/resource for which to retrieve property names.
     *
     * @return An unordered collection of property names, or an empty collection if no property names were found.
     *
     * @throws RepositoryException If an error occurs accessing the user properties.
     */
    Collection<String> getPropertyNames(String relPath) throws RepositoryException;

    /**
     * Retrieves the value of a property found on one or more of the user properties contained in this composite. The
     * property value found on the last user properties takes precedence. The order of the user properties is governed
     * by the order of the relative paths if the composite was retrieved via {@link
     * UserPropertiesManager#getUserPropertiesComposite(String, String[])}, or via their natural ordering within the
     * user node tree if retrieved via {@link UserPropertiesManager#getUserPropertiesComposite(String,
     * UserPropertiesFilter)}.
     *
     * @param relativePath The relative path of the property to retrieve.
     *
     * @return The value of the property, or <code>null</code> if no such property was found.
     *
     * @throws RepositoryException If an error occurs accessing the user properties.
     */
    String getProperty(String relativePath) throws RepositoryException;

    /**
     * Same as {@link #getProperty(String)}. Additionally a default value and type can be specified.
     *
     * @param relativePath The relative path of the property to retrieve.
     * @param defaultValue The default value to return if the property is not found.
     * @param type         The expected type of the property value.
     * @param <T>          specifies the expected format of the property
     * 
     * @return The property value found at <code>relativePath</code> with the given <code>type</code>, or the
     *         <code>defaultValue</code> if the property is not found.
     *
     * @throws RepositoryException If an error occurs accessing the user properties.
     */
    <T> T getProperty(String relativePath, T defaultValue, Class<T> type)
            throws RepositoryException;

    /**
     * Returns the user properties that make up this composite.
     *
     * @return An unordered collection of user properties.
     */
    Collection<UserProperties> getUserProperties();

    /**
     * Returns the ID of the authorizable to which the user properties of this composite belong.
     *
     * @return A string representing the authorizable ID.
     */
    String getAuthorizableId();

    /**
     * Returns the Collection of paths of the aggregated {@link UserProperties}
     *
     * @return A collection of paths
     * @throws RepositoryException throws on error
     */
    Collection<String> getUserPropertiesPaths() throws RepositoryException;
}
