/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.security.user;

import java.security.Principal;
import java.util.Comparator;
import java.util.Iterator;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;

import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.jackrabbit.api.security.user.Group;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.commons.osgi.PropertiesUtil;

/**
 * The <code>UserPropertiesManager</code> interface provides access to
 * <code>UserProperties</code> associated with a given authorizable.
 */
@ProviderType
public interface UserPropertiesManager {

    /**
     * Constant for the UserProperties name.
     */
    String JCR_TITLE = Property.JCR_TITLE;

    /**
     * Constant for the ranking property name.
     */
    String GRANITE_RANKING = "granite:ranking";

    /**
     * Constant for the ranking property.
     */
    int DEFAULT_RANKING = 1000;


    /**
     * Comparator used to sort UserProperties by descending ranking.
     */
    Comparator<Node> DESCENDING_RANKING_COMPARATOR = new Comparator<Node>() {
        @Override
        public int compare(final Node p1, final Node p2) {
            try {
                return ranking(p1)- ranking(p2);
            } catch (RepositoryException e) {
                return 0; // nodes are considered equal
            }
        }

        private int ranking(final Node p) throws RepositoryException {
            if(p.hasProperty(GRANITE_RANKING)) {
                return PropertiesUtil.toInteger(p.getProperty(GRANITE_RANKING).getLong(), DEFAULT_RANKING);
            }
            return DEFAULT_RANKING;
        }
    };

    /**
     * Create a new instance of <code>UserProperties</code>. Please note that
     * {@link javax.jcr.Session#save()} must be called by the editing JCR
     * session in order to persist the new user properties.
     *
     * @param authorizableId The ID of the associated authorizable.
     * @param relPath A name or relative path identifying the new user properties.
     * @return A new <code>UserProperties</code> instance. Note that the
     * associated JCR node will be {@link javax.jcr.Item#isNew() NEW} until
     * {@link javax.jcr.Session#save() Session.save} is called to persist the changes.
     * @throws javax.jcr.RepositoryException If an error occurs or the specified
     * relative path is invalid.
     */
    UserProperties createUserProperties(String authorizableId, String relPath) throws RepositoryException;

    /**
     * Retrieve the user properties identified by the given relative path and
     * the specified authorizable ID.
     *
     * @param authorizableId Identifier of the target authorizable.
     * @param relPath A name or relative path identifying the user properties.
     * @return An instance of <code>UserProperties</code> or <code>null</code>
     * if the authorizable does not exist or it has no user properties at the
     * specified relative path.
     * @throws RepositoryException If an error occurs or the specified
     * relative path is invalid.
     */
    UserProperties getUserProperties(String authorizableId, String relPath) throws RepositoryException;

    /**
     * Retrieve the user properties identified by the given relative path and
     * the specified authorizable.
     *
     * @param authorizable The target authorizable.
     * @param relPath A name or relative path identifying the user properties.
     * @return An instance of <code>UserProperties</code> or <code>null</code>
     * if the authorizable has no user properties at the specified relative path.
     * @throws RepositoryException If an error occurs or the specified
     * relative path is invalid.
     */
    UserProperties getUserProperties(Authorizable authorizable, String relPath) throws RepositoryException;

    /**
     * Retrieve the user properties identified by the given node.
     *
     * @param userPropertiesNode The node representing the user properties.
     * @return The user properties identified by the given node.
     * @throws RepositoryException If an exception occurs or if the given node
     * does not represent valid user properties (e.g. not associated with an
     * authorizable).
     */
    UserProperties getUserProperties(Node userPropertiesNode) throws RepositoryException;

    /**
     * Retrieves a composite (aggregation) of {@link UserProperties} associated with the authorizable identified by the
     * given <code>authorizableId</code> and corresponding to  and in the order of the given <code>relPaths</code>. The
     * ordering is relevant for e.g. accessing a property found on multiple user property nodes, in which case it is
     * inherited according to the order (last has precedence).
     *
     * @param authorizableId The ID of the {@link Authorizable} for which to retrieve the user properties composite.
     * @param relPaths       An array of strings denoting the desired relative paths of the user property nodes and
     *                       their order in which to be included in the composite.
     *
     * @return A composite of {@link UserProperties}.
     *
     * @throws RepositoryException If an error occurs during repository access or no authorizable was found with the
     *                             given <code>authorizableId</code>.
     */
    UserPropertiesComposite getUserPropertiesComposite(String authorizableId, String[] relPaths)
            throws RepositoryException;

    /**
     * Retrieves a composite (aggregation) of {@link UserProperties} associated with the authorizable identified by the
     * given <code>authorizableId</code> and included by the given <code>filter</code>.
     *
     * @param authorizableId The ID of the {@link Authorizable} for which to retrieve the user properties composite.
     * @param filter         A {@link UserPropertiesFilter} by which user properties are included or excluded.
     *
     * @return A {@link UserPropertiesComposite} containing the {@link UserProperties} included by the filter.
     *
     * @throws RepositoryException If an error occurs during repository access or no authorizable was found with the
     *                             given <code>authorizableId</code>.
     */
    UserPropertiesComposite getUserPropertiesComposite(String authorizableId, UserPropertiesFilter filter)
            throws RepositoryException;

    /**
     * Retrieves a composite (aggregation) of {@link UserProperties} associated with the authorizable identified by the
     * given <code>authorizableId</code>. The {@link UserProperties} are retrieved with the {@link javax.jcr.Session}
     * passed to the @{link UserPropertiesManager}, then are ordered by descending ranking using
     * {@link UserPropertiesManager#DESCENDING_RANKING_COMPARATOR} prior to being aggregated. The table below illustrates this behavior.
     *
     * <table summary="">
     * <thead>
     * <tr>
     * <th></th>
     * <th>Profile 1 value</th>
     * <th>Profile 2 value</th>
     * <th>Profile 3 value</th>
     * <th>Value returns by UserPropertiesComposite</th>
     * </tr>
     * </thead>
     * <tbody>
     * <tr>
     * <td>Is Readable by user session</td>
     * <td>No</td>
     * <td>Yes</td>
     * <td>Yes</td>
     * <td>-</td>
     * </tr>
     * <tr>
     * <td>Ranking</td>
     * <td>700</td>
     * <td>500</td>
     * <td>100</td>
     * <td>-</td>
     * </tr>
     * <tr>
     * <td>Name</td>
     * <td>Robert Jnr</td>
     * <td>Robert</td>
     * <td>Bob</td>
     * <td>Robert</td>
     * </tr>
     * <tr>
     * <td>Age</td>
     * <td>(undefined)</td>
     * <td>(undefined)</td>
     * <td>25</td>
     * <td>25</td>
     * </tr>
     * <tr>
     * <td>email</td>
     * <td>robert.private@example.com</td>
     * <td>(undefined)</td>
     * <td>(undefined)</td>
     * <td>(undefined)</td>
     * </tr>
     * </tbody>
     * </table>
     *
     * @param authorizableId The ID of the {@link Authorizable} for which to retrieve the user properties composite.
     * @param relRootPath relative root path of the {@link UserProperties} to be aggregated
     * @return A {@link UserPropertiesComposite} containing the {@link UserProperties}
     * @throws RepositoryException If an error occurs during repository access or no authorizable was found with the
     *                             given <code>authorizableId</code>.
     */
    @Nonnull
    UserPropertiesComposite getUserPropertiesComposite(@Nonnull final String authorizableId, @Nullable String relRootPath) throws RepositoryException;

    /**
     * Retrieves with the given session a composite (aggregation) of {@link UserProperties} associated with
     * the authorizable identified by the given <code>authorizableId</code>.
     * The {@link UserProperties} are ordered using the given {@link Comparator} prior to be aggregated.
     * The {@link UserPropertiesComposite} will contain the most private properties accessible by the user session.
     *
     * See example in {@link UserPropertiesManager#getUserPropertiesComposite(String, String)}.
     *
     * @param authorizableId The ID of the {@link Authorizable} for which to retrieve the user properties composite.
     * @param relRootPath relative root path
     * @param comparator The comparator used to sort candidates in a specific order
     * @return A {@link UserPropertiesComposite} containing the {@link UserProperties}.
     *
     * @throws RepositoryException If an error occurs during repository access or no authorizable was found with the
     *                             given <code>authorizableId</code>.
     */
    @Nonnull
    UserPropertiesComposite getUserPropertiesComposite(@Nonnull String authorizableId, @Nullable String relRootPath, @Nonnull Comparator<Node> comparator) throws RepositoryException;

    /**
     * Returns an iterator over the properties of all groups that the authorizable with <code>authorizableId</code> is
     * a member of. please note that only groups are included of which the underlying session has read access to.
     *
     * @param authorizableId the id of the authorizable
     * @param relPath the relative path to build the user properties from
     * @param declaredOnly if {@code true} only declared groups are included
     * @return an iterator of user properties
     * @throws RepositoryException if an error occurrs
     *
     * @see org.apache.jackrabbit.api.security.user.Authorizable#memberOf()
     * @see org.apache.jackrabbit.api.security.user.Authorizable#declaredMemberOf()
     */
    Iterator<UserProperties> getMemberOfUserProperties(String authorizableId, String relPath, boolean declaredOnly)
            throws RepositoryException;

    /**
     * Returns an iterator over the user properties of all members of the given group.
     *
     * @param group the group to retrieve the members from
     * @param relPath the relative path to build the user properties from
     * @param declaredOnly if {@code true} only delcated members are included
     * @return an iterator of user properties
     * @throws RepositoryException if an error occurrs
     *
     * @see org.apache.jackrabbit.api.security.user.Group#getMembers()
     * @see org.apache.jackrabbit.api.security.user.Group#getDeclaredMembers()
     */
    Iterator<UserProperties> getMemberUserProperties(Group group, String relPath, boolean declaredOnly)
            throws RepositoryException;

    /**
     * Grant read permission on the UserProperties to the given Principal.
     * The permissions are set using the session provided while obtaining the {@link UserPropertiesManager} instance.
     *
     * Principal can be retrieved with {@link org.apache.jackrabbit.api.security.principal.PrincipalManager#getPrincipal(String)}.
     * Similarly Everyone Principal can be retrieved with {@link org.apache.jackrabbit.api.security.principal.PrincipalManager#getEveryone()}.
     *
     * @param userProperties user properties to which readers should be added
     * @param principals Principal to which read access must be granted
     * @return true if permissions have been modified
     * @throws RepositoryException if an error occurs or the editing session is not allowed to modify access control
     */
    boolean addReaders(@Nonnull UserProperties userProperties, @Nonnull Principal... principals) throws RepositoryException;

    /**
     * Remove read permission granted on the UserProperties to the given Principal.
     * The permissions are updated using the session provided while obtaining the {@link UserPropertiesManager} instance.
     *
     * Note that the Authorizable may keep having read access to the UserProperties if inherited from group membership
     * or glob permissions.
     *
     * @param userProperties user properties to which readers should be removed
     * @param principals Principal to which read access must be revoked
     * @return true if permissions have been modified
     * @throws RepositoryException if an error occurs or the editing session is not allowed to modify access control
     */
    boolean removeReaders(@Nonnull UserProperties userProperties, @Nonnull Principal... principals) throws RepositoryException;

}
