/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.taskmanagement;

import java.util.HashMap;

/**
 * Enumerates the built in task properties
 */
public enum TaskProperty {

    /**
     * The task name
     */
    NAME("name"),

    /**
     * The task description
     */
    DESCRIPTION("description"),

    /**
     * The task instructions
     */
    INSTRUCTIONS("instructions"),

    /**
     * The name of the principal who owns the task
     */
    OWNER_ID("assignee"),

    /**
     * The task type
     */
    TASK_TYPE_NAME("taskTypeName"),

    /**
     * The task status
     */
    STATUS("status"),

    /**
     * The time the task was created
     */
    CREATED_TIME("startTime"),

    /**
     * The Id of the user who created this task.  In the case where this task was created by the system this field will be empty
     */
    CREATED_BY("createdBy"),

    /**
     * The time the task was last modified
     */
    MODIFIED_TIME("lastModified"),

    /**
     * The id for the user who last modified the task
     */
    MODIFIED_BY("lastModifiedBy"),

    /**
     * When the task was completed
     */
    COMPLETED_TIME("endTime"),

    /**
     * The id for the user who completed the task
     */
    COMPLETED_BY("completedBy"),

    /**
     * The action names of the task
     */
    ACTION_NAMES("actionNames"),

    /**
     * The action used to complete the task
     */
    SELECTED_ACTION("selectedAction"),

    /**
     * The Id of the parent task
     */
    PARENT_TASK_ID("parentTaskId"),

    /**
     * A content path the task is associated to
     */
    CONTENT_PATH("contentPath"),

    /**
     * name hierarchy for this task (co-insides with parent task id hierarchy)
     */
    NAME_HIERARCHY("nameHierarchy");


    private static HashMap<String, TaskProperty> PROPERTIES_BY_ID;

    /**
     * Used to map the property name to the enum value and define the intrinsic 
     * property name
     */
    static{
        PROPERTIES_BY_ID = new HashMap<String, TaskProperty>(TaskProperty.values().length);
        for (TaskProperty _prop : TaskProperty.values()){
            PROPERTIES_BY_ID.put(_prop.getPropertyName(), _prop);
        }
    }

    /**
     * The property name for this instance
     */
    private String propertyName;
    
    /**
     * Create a new IntrinsicProperty and specify the property name
     * 
     * @param propertyName The property name to set
     */
    private TaskProperty(String propertyName){
        this.propertyName = propertyName;
    }

    /**
     * Return the property name which can be used in task filter conditions.
     * see {@link com.adobe.granite.taskmanagement.TaskManager#getTasks(com.adobe.granite.taskmanagement.Filter)}
     * see {@link com.adobe.granite.taskmanagement.Condition#setPropertyName(String)}
     * 
     * @return A String representing the property name of this instance
     * or null if it has not been set
     */
    public String getPropertyName(){
        return propertyName;
    }

    /**
     * Returns true if this is an intrinsic property (ie, one of DESCRIPTION, INSTRUCTIONS, ... , ACTION_NAMES) and false otherwise.
     * 
     * @param propertyName  The property name to check
     * @return true if an intrinsic property and false otherwise
     */
    public static final boolean isTaskProperty(String propertyName){
        return PROPERTIES_BY_ID.containsKey(propertyName);
    }

    /**
     * Return the TaskProperty object for the specified property name
     * 
     * @param propertyName used to specify the property name
     * @return an TaskProperty object for the specified property name
     * and false if the property name is not an intrinsic property.
     */
    public static final TaskProperty getTaskProperty(String propertyName){
        return PROPERTIES_BY_ID.get(propertyName);
    }
}
