/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.translation.api;

//
/*
 * The TranslationException class extends the Exception class and adds an ErrorCode enum.  This ErrorCode contains
 * the specific type of failure that occurred within the Translation Framework API call.  Please see the ErrorCode
 * section for details on specific ErrorCode types.
 */
/** The Class TranslationException. */
public class TranslationException extends Exception {

    /**
     *
     */
    private static final long serialVersionUID = -1281094183029567014L;

    /** The error code. */
    private ErrorCode errorCode = null;

    /** The message. */
    private String message = null;

    /**
     * Instantiates a new translation exception.
     * @param message the message
     * @param errorCode the error code
     */
    public TranslationException(final String message, final ErrorCode errorCode) {
        super(message);
        this.message = message;
        this.errorCode = errorCode;
    }

    /**
     * Instantiates a new translation exception.
     * @param message the message
     * @param throwable the throwable
     * @param errorCode the error code
     */
    public TranslationException(final String message, final Throwable throwable, final ErrorCode errorCode) {
        super(message, throwable);
        this.message = message;
        this.errorCode = errorCode;
    }

    /*
     * This method overrides the base Exception toString method and outputs the ErrorCode in conjunction with the
     * Exception message.
     */
    /*
     * (non-Javadoc)
     * @see java.lang.Throwable#toString()
     */
    @Override
    public String toString() {
        if (errorCode != null && message != null) {
            return "Error Code: " + errorCode + ".  Message: " + message;
        } else if (message != null) {
            return message;
        } else {
            return super.toString();
        }
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Throwable#getMessage()
     */
    @Override
    public String getMessage() {
        return message;
    }

    /**
     * Gets the error code.
     * @return the error code
     */
    public ErrorCode getErrorCode() {
        return errorCode;
    }

    /*
     * This enum provides specific error codes for handling and throwing exceptions in the Translation API. It is
     * expected that Connectors implementing the Translation API will leverage these constants when specific
     * exceptions occur. If an exception is of an unknown type the GENERAL_EXCEPTION will be thrown.
     */
    /** The Enum ErrorCode. */
    public enum ErrorCode {

        /*
         * Thrown for all exceptions that do not fit other exception error code types
         */
        /** The general exception. */
        GENERAL_EXCEPTION,

        /*
         * An exception occurred during translation
         */
        /** The translation failed. */
        TRANSLATION_FAILED,

        /*
         * The request to the translation engine failed
         */
        /** The request failed. */
        REQUEST_FAILED,

        /*
         * The MT engine specified by this Connector is unavailable/unknown
         */
        /** The no engine. */
        NO_ENGINE,

        /*
         * The content format specified is not supported by this MT engine
         */
        /** The not supported format. */
        NOT_SUPPORTED_FORMAT,

        /*
         * The request has timed out
         */
        /** The request timeout. */
        REQUEST_TIMEOUT,

        /*
         * Parameters necessary for this functionality are missing
         */
        /** The missing parameter. */
        MISSING_PARAMETER,

        /*
         * The request to the translation engine was interrupted
         */
        /** The request interrupt. */
        REQUEST_INTERRUPT,

        /*
         * The MT engine is temporarily unavailable
         */
        /** The engine temp unavailable. */
        ENGINE_TEMP_UNAVAILABLE,

        /*
         * To be used for exceptions while detecting a language of a string.
         */
        /** The detection failure. */
        DETECTION_FAILURE,

        /*
         * Thrown if a call to the translation function has the same to and from language.
         */
        /** The to from same language. */
        TO_FROM_SAME_LANGUAGE,

        /*
         * Returned when a language cannot be determined by the "detect" function
         */
        /** The unknown language. */
        UNKNOWN_LANGUAGE,

        /*
         * Thrown when the functionality called is not implemented in this connector
         */
        /** The service not implemented. */
        SERVICE_NOT_IMPLEMENTED,

        /*
         * Thrown when the connector doesn't support this language translation direction
         */
        /** The not supported lang direction. */
        NOT_SUPPORTED_LANG_DIRECTION,

        /*
         * Language code passed in is not in a valid format or not part of the supported list
         */
        /** The invalid language. */
        INVALID_LANGUAGE,

        /*
         * There are no registered Translation Service Factories in this system
         */
        /** The no registered factories. */
        NO_REGISTERED_FACTORIES,

        /*
         * There are no registered Translation Service Factories for this name
         */
        /** The unknown factory name. */
        UNKNOWN_FACTORY_NAME,

        /*
         * The Cloud Configuration credentials are blank or missing
         */
        /** The missing credentials. */
        MISSING_CREDENTIALS,

        /*
         * Language code passed in is "valid" but not supported by this current connector
         */
        /** The unsupported language. */
        UNSUPPORTED_LANGUAGE,

        /*
         * Current credentials are not authorized for this functionality
         */
        /** The not authorized. */
        NOT_AUTHORIZED

    }
}