/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.translation.api;

import java.io.InputStream;
import java.util.Iterator;
import java.util.zip.ZipInputStream;

import aQute.bnd.annotation.ProviderType;
import com.adobe.granite.comments.Comment;
import com.adobe.granite.comments.CommentCollection;

//
/** The Interface TranslationObject. */
@ProviderType
public interface TranslationObject {

    /**
     * Gets the title of the TranslationObject. Title is auto-populated from TranslationObject.
     * @return the title
     */
    String getTitle();

    /**
     * Gets the path to the Translation Object being translated. The path returned is to where the target Translation
     * Object is stored in AEM CRX de. The path is derived by coping that of the the source Translation Object.
     * @return the content url
     */
    String getTranslationObjectTargetPath();

    /**
     * Gets the path to the source Translation Object. The path returned is to the where the source Translation Object
     * is stored in AEM CRX de.
     * @return the source content url
     */
    String getTranslationObjectSourcePath();

    /**
     * Gets the version name of the source Object. If this name is null, then latest version of source content is
     * used.
     * @return the version name
     */
    String getSourceVersion();

    /**
     * Gets the mime type of TranslationObject.
     * @return the mime type
     */
    String getMimeType();

    /**
     * Gets confirmation return from the external translation service provider that they have received the
     * TranslationObject for translation. Translation service provider returns a unique TranslationObjectId.
     * @return Translation Object Id
     */
    String getId();

    /**
     * Gets the comment collection. This returns all the comments added to the TranslationObject.
     * @return the comment Collection
     */
    CommentCollection<Comment> getCommentCollection();

    /**
     * Gets the supporting translation objects iterator. Each TranslationObject can have a list of supporting
     * translation objects which add reference to the TranslationObject.
     * @return the supporting translation objects iterator
     */
    Iterator<TranslationObject> getSupportingTranslationObjectsIterator();

    /**
     * Gets the supporting translation objects count.
     * @return the supporting translation objects count
     */
    int getSupportingTranslationObjectsCount();

    /**
     * Gets the Translation Job metadata
     * @return the Translation Job metadata
     */
    TranslationMetadata getTranslationJobMetadata();

    /**
     * @deprecated Use {@link #getTranslationObjectXMLInputStream()} instead.
     * Gets XML input stream for the Translation Object to be translated.
     * @return the translation Object input stream
     */
    @Deprecated
    InputStream getTranslationObjectInputStream() throws TranslationException;

    /**
     * Gets XML input stream for the Translation Object to be translated.
     * @return The translation Object XML input stream
     */
    InputStream getTranslationObjectXMLInputStream() throws TranslationException;

    /**
     * Gets XLIFF input stream for the Translation Object to be translated.
     * @param xliffVersion Requested version of XLIFF
     * @return The translation Object XLIFF input stream
     */
    InputStream getTranslationObjectXLIFFInputStream(String xliffVersion) throws TranslationException;

    /**
     * Gets the translated Object input stream. Returns null in case TranslationObject was not translated.
     * @return the translated Object input stream
     */
    InputStream getTranslatedObjectInputStream();

    /**
     * Gets a zip input stream containing the preview of Translation Object. This zip contains all the files required
     * for offline preview of the translation object.
     * @return Zip input stream, containing the preview of Translation Object.
     */
    ZipInputStream getTranslationObjectPreview();
}