/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.ui.clientlibs;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.Map;

import org.apache.sling.api.SlingHttpServletRequest;

import aQute.bnd.annotation.ProviderType;


/**
 * <code>HtmlLibraryManager</code> provides access to repository defined
 * html libraries.
 */
@ProviderType
public interface HtmlLibraryManager {

    /**
     * request parameter name for enabling debug console (firebug + cq logging)
     */
    final String PARAM_DEBUG_CONSOLE = "debugConsole";

    /**
     * request parameter name for turning on HtmlLibraryServlet debugging
     */
    final String PARAM_DEBUG_CLIENT_LIBS = "debugClientLibs";

    /**
     * request parameter name for testing a theme
     */
    final String PARAM_FORCE_THEME = "forceTheme";

    /**
     * Request attribute flag that forces the inclusion of the CQURLInfo
     * @since 5.5.42
     */
    final String REQUEST_ATTR_FORCE_CQ_URLINFO = "com.day.cq.widget.htmllibrarymanager.forceurlinfo";

    /**
     * Writes the JS include snippets to the given writer. The paths to the
     * JS libraries are included that match the given categories. Note that
     * themed and non-themed libraries are included. If the request contains
     * a {@value #PARAM_FORCE_THEME} parameter, the themed libraries are
     * overlaid with their respective counterparts with that given theme.
     *
     * @param request request
     * @param out writer
     * @param categories categories
     * @throws java.io.IOException if an I/O error occurs
     */
    void writeJsInclude(SlingHttpServletRequest request, Writer out, String... categories)
            throws IOException;

    /**
     * Writes the JS include snippets to the given writer. The paths to the
     * JS libraries are included that match the given categories.
     * If <code>themed</code> is <code>false</code>, only non-themed
     * libraries are included.
     * If <code>themed</code> is <code>true</code> only themed libraries are
     * included and if the request contains a {@value #PARAM_FORCE_THEME}
     * parameter, the themed libraries are overlaid with their respective
     * counterparts with that given theme.
     *
     * @param request request
     * @param out writer
     * @param themed controls if themed or non themed libraries should be included
     * @param categories categories
     * @throws java.io.IOException if an I/O error occurs
     *
     * @since 5.4
     */
    void writeJsInclude(SlingHttpServletRequest request, Writer out,
                        boolean themed, String... categories)
            throws IOException;

    /**
     * Writes the CSS include snippets to the given writer. The paths to the
     * CSS libraries are included that match the given categories. Note that
     * themed and non-themed libraries are included. If the request contains
     * a {@value #PARAM_FORCE_THEME} parameter, the themed libraries are
     * overlaid with their respective counterparts with that given theme.
     *
     * @param request request
     * @param out writer
     * @param categories categories
     * @throws java.io.IOException if an I/O error occurs
     */
    void writeCssInclude(SlingHttpServletRequest request, Writer out, String... categories)
            throws IOException;

    /**
     * Writes the CSS include snippets to the given writer. The paths to the
     * CSS libraries are included that match the given categories.
     * If <code>themed</code> is <code>false</code>, only non-themed
     * libraries are included.
     * If <code>themed</code> is <code>true</code> only themed libraries are
     * included and if the request contains a {@value #PARAM_FORCE_THEME}
     * parameter, the themed libraries are overlaid with their respective
     * counterparts with that given theme.
     *
     * @param request request
     * @param out writer
     * @param themed controls if themed or non themed libraries should be included
     * @param categories categories
     * @throws java.io.IOException if an I/O error occurs
     *
     * @since 5.4
     */
    void writeCssInclude(SlingHttpServletRequest request, Writer out,
                         boolean themed, String... categories)
            throws IOException;

    /**
     * Writes all CSS and only themed JS include snippets to the given writer.
     * The paths to the libraries are included that match the given categories.
     * If the request contains a {@value #PARAM_FORCE_THEME} parameter, the
     * themed libraries are overlaid with their respective counterparts with
     * that given theme.
     *
     * Please note, that the theme include should happen after the js includes
     * since the theme js might reference themes.
     *
     * @param request request
     * @param out writer
     * @param categories categories
     * @throws java.io.IOException if an I/O error occurs
     */
    void writeThemeInclude(SlingHttpServletRequest request, Writer out, String... categories)
        throws IOException;

    /**
     * Writes the include snippets to the given writer. The paths to the
     * libraries are included that match the given categories and the theme
     * name that is extracted from the request.<br>
     *
     * Same as:<br>
     * <code>
     * writeCssInclude(...);
     * writeJsInclude(...);
     * writeThemeInclude(...);
     * </code>
     *
     * If one of the libraries to be included has assigned channels, then the
     * inclusion is delegated to the client side library manager.
     * 
     * @param request request
     * @param out writer
     * @param categories categories
     * @throws java.io.IOException if an I/O error occurs
     */
    void writeIncludes(SlingHttpServletRequest request, Writer out, String... categories)
        throws IOException;

    /**
     * Returns the html library that is configured at the given path. If no
     * such library exists, <code>null</code> is returned.
     *
     * @param type the library type
     * @param path the path
     * @return the library or null
     */
    HtmlLibrary getLibrary(LibraryType type, String path);

    /**
     * Returns the html library that is address by the given request. if no
     * such library exists, <code>null</code> is returned.
     *
     * @param request the request
     * @return the library or null
     */
    HtmlLibrary getLibrary(SlingHttpServletRequest request);

    /**
     * Checks if library minification is enabled.
     * @return <code>true</code> if minification is enabled.
     */
    boolean isMinifyEnabled();

    /**
     * Checks if debug support is enabled.
     * @return <code>true</code> if debug is enabled.
     */
    boolean isDebugEnabled();

    /**
     * Checks if gzip compression is enabled.
     * @return <code>true</code> if gzip is enabled.
     */
    boolean isGzipEnabled();

    /**
     * Returns all client libraries
     * @return all client libraries
     *
     * @since 5.4
     */
    Map<String, ClientLibrary> getLibraries();

    /**
     * Returns all client libraries that match the specified filters. If a theme
     * name is specified, only themed libraries are returned, otherwise only
     * non-theme libraries are returned. If theme name is an empty string,
     * the configured default theme is used.
     *
     * @param categories the categories
     * @param type type or <code>null</code> to match all types
     * @param ignoreThemed <code>true</code> to filter out themed libraries
     * @param transitive <code>true</code> to resolve recursively
     * @return matching client libraries
     *
     * @since 5.4
     */
    Collection<ClientLibrary> getLibraries(String[] categories,
                                           LibraryType type,
                                           boolean ignoreThemed,
                                           boolean transitive);
    /**
     * Returns all themed client libraries that match the type. If type is
     * <code>null</code> all libraries are returned. If theme name is an empty
     * string, the configured default theme is used. If theme name is <code>null</code>
     * all themed libraries are returned.
     *
     * @param categories the categories
     * @param type type or <code>null</code> to match all types
     * @param themeName theme name or <code>null</code>
     * @param transitive <code>true</code> to resolve recursively
     * @return matching client libraries
     *
     * @since 5.4
     */
    Collection<ClientLibrary> getThemeLibraries(String[] categories,
                                                LibraryType type,
                                                String themeName,
                                                boolean transitive);

}