/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.ui.components;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.CompositeValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;

/**
 * The config properties of a resource.
 */
public class Config {
    private Resource resource;

    /**
     * Current resource properties merged with default properties.
     */
    private ValueMap properties;

    /**
     * Current resource properties merge with parent's properties.
     */
    private ValueMap propertiesWithParent;

    private Resource parentResource;

    /**
     * The resource name for the defaults config ("defaults").
     */
    public static String DEFAULTS = "defaults";

    /**
     * The resource name for items ("items").
     */
    public static String ITEMS = "items";

    /**
     * The resource name for layout config ("layout").
     */
    public static String LAYOUT = "layout";

    /**
     * The resource name for datasource config ("datasource").
     */
    public static String DATASOURCE = "datasource";

    /**
     * The resource name for render condition config ("rendercondition").
     */
    public static String RENDERCONDITION = "rendercondition";

    /**
     * Alias of {@link #Config(Resource, boolean)} with inherit is
     * <code>false</code>.
     * @param resource
     *            The resource
     */
    public Config(Resource resource) {
        this(resource, false);
    }

    /**
     * Create a new Config object for the given resource. If
     * <code>inherit</code> is <code>true</code>, the default properties are
     * setup and taken from the parent's {@link #DEFAULTS} sub-resource, and
     * parent properties are used when calling
     * {@link #getInherited(String, Class)}.
     * 
     * @param resource
     *            The resource
     * @param inherit
     *            Whether to inherit default properties or not
     * @see #getParentResource()
     */
    public Config(Resource resource, boolean inherit) {
        this.resource = resource;
        properties = resource != null ? resource.getValueMap() : new ValueMapDecorator(new HashMap<String, Object>());

        if (inherit) {
            parentResource = getParentResource();

            if (parentResource != null) {
                Resource defaults = parentResource.getChild(DEFAULTS);
                if (defaults != null) {
                    properties = new CompositeValueMap(properties, defaults.getValueMap());
                }
                
                propertiesWithParent = new CompositeValueMap(properties, parentResource.getValueMap());
            }
        }
    }

    /**
     * Create a new Config object for the given resource.
     * 
     * @param resource
     *            The resource
     * @param defaults
     *            The default properties. Pass <code>null</code> if these are
     *            not required.
     * @param parentProperties
     *            The properties of the parent resource. Pass <code>null</code>
     *            if these are not required.
     */
    public Config(Resource resource, ValueMap defaults, ValueMap parentProperties) {
        this.resource = resource;

        ValueMap prop = resource != null ? resource.getValueMap() : new ValueMapDecorator(new HashMap<String, Object>());

        properties = new CompositeValueMap(prop, defaults);
        propertiesWithParent = new CompositeValueMap(prop, parentProperties);
    }

    /**
     * Returns the value of property with the given name, converted to string.
     * If the property does not exist it is taken from the defaults. Return an
     * empty string if still not existing can't be converted.
     * @param name the name of the property
     * @return the value of property with the given name, converted to string
     */
    public String get(String name) {
        return get(name, "");
    }

    /**
     * Returns the value of property with the given name, converted into the
     * given type. If the property does not exist it is taken from the defaults.
     * Return the given defaultValue if still non existing or can't be
     * converted.
     * 
     * @param name
     *            The name of the property
     * @param defaultValue
     *            The default value to use if the named property does not exist
     *            or cannot be converted to the requested type. The default
     *            value is also used to define the type to convert the value to.
     *            If this is <code>null</code> any existing property is not
     *            converted.
     * @param <T>
     *            The type of the default value
     * @return
     *            The value of property with the given name, converted into the
     *            given type
     */
    public <T> T get(String name, T defaultValue) {
        return properties.get(name, defaultValue);
    }

    /**
     * Returns the value of property with the given name, converted into the
     * given type. If the property does not exist, it is taken from the
     * defaults. Return <code>null</code> if non existing or can't be converted.
     * 
     * @param name
     *            The name of the property
     * @param type
     *            The type to convert the value of the given property name to
     * @param <T>
     *            The type of the submitted type param
     * @return
     *            The value of property with the given name, converted into the
     *            given type
     */
    public <T> T get(String name, Class<T> type) {
        return properties.get(name, type);
    }

    /**
     * Returns the value of property with the given name, converted to string.
     * If the property does not exist, it is taken from the parent properties.
     * Return an empty string if still not existing can't be converted.
     * 
     * @param name
     *            The name of the property
     * @return
     *            The value of property with the given name, converted to string.
     * 
     * @see #getParentResource()
     */
    public String getInherited(String name) {
        return getInherited(name, "");
    }

    /**
     * Returns the value of property with the given name, converted into the
     * given type. If the property does not exist, it is taken from the parent
     * properties. Return the given defaultValue if still non existing or can't
     * be converted.
     * 
     * @param name
     *            The name of the property
     * @param defaultValue
     *            The default value to use if the named property does not exist
     *            or cannot be converted to the requested type. The default
     *            value is also used to define the type to convert the value to.
     *            If this is <code>null</code> any existing property is not
     *            converted.
     * @param <T>
     *            The type to convert the value of the property to
     * @return
     *            The value of property with the given name, converted into the
     *            given type
     * 
     * @see #getParentResource()
     */
    public <T> T getInherited(String name, T defaultValue) {
        return propertiesWithParent.get(name, defaultValue);
    }

    /**
     * Returns the value of property with the given name, converted into the
     * given type. If the property does not exist, it is taken from the parent
     * properties. Return <code>null</code> if non existing or can't be
     * converted.
     * 
     * @param name
     *            The name of the property
     * @param type
     *            The type to convert the value of the given property name to
     * @param <T>
     *            The type to convert the value of the property to
     * @return
     *            The value of property with the given name, converted into the
     *            given type
     * 
     * @see #getParentResource()
     */
    public <T> T getInherited(String name, Class<T> type) {
        return propertiesWithParent.get(name, type);
    }

    /**
     * Alias of {@link #get(String)}.
     * @param name
     *            The name of the property
     * @return
     *            The value of property with the given name, converted to string
     */
    public String getInheritedDefault(String name) {
        return getInheritedDefault(name, "");
    }

    /**
     * Alias of {@link #get(String, Object)};
     * @return
     *            The value of property with the given name, converted to string
     * @param name
     *            The name of the property
     * @param defaultValue
     *            The default value to use if the named property does not exist
     *            or cannot be converted to the requested type. The default
     *            value is also used to define the type to convert the value to.
     *            If this is <code>null</code> any existing property is not
     *            converted.
     * @param <T>
     *            The type to convert the value of the property to
     */
    public <T> T getInheritedDefault(String name, T defaultValue) {
        return get(name, defaultValue);
    }

    /**
     * Alias of {@link #get(String, Class)}.
     * @return
     *            The value of property with the given name, converted to string
     * @param name
     *            The name of the property
     * @param type
     *            The type to convert the value of the given property name to
     * @param <T>
     *            The type to convert the value of the property to
     */
    public <T> T getInheritedDefault(String name, Class<T> type) {
        return get(name, type);
    }

    /**
     * Returns the default properties that will be applied to child elements. Be
     * aware that these are not the defaults that are applied to the resource
     * itself.
     * @return
     *            The value map containing the default properties
     */
    public ValueMap getDefaultProperties() {
        Resource defaults = resource.getChild(DEFAULTS);
        return defaults != null ? defaults.getValueMap() : new ValueMapDecorator(new HashMap<String, Object>());
    }

    /**
     * Returns the properties of the resource merged with the default
     * properties.
     * @return
     *            The value map containing the properties merged with the default
     *            properties. 
     */
    public ValueMap getProperties() {
        return properties;
    }

    /**
     * Returns the child resources of {@link #ITEMS} sub-resource.
     * @return
     *            The child resources of the specified {@link #ITEMS} sub-resource  
     */
    public Iterator<Resource> getItems() {
        return getItems(resource, ITEMS);
    }

    /**
     * Returns the child resources of the sub-resource with the given name.
     * @param name
     *            The name of the resource
     * @return
     *            The child resources of the sub-resource with the given name
     */
    public Iterator<Resource> getItems(String name) {
        return getItems(resource, name);
    }

    /**
     * Returns the child resources of {@link #ITEMS} sub-resource of the given
     * resource.
     * @param resource
     *            The resource
     * @return
     *            The child resources of the {@link #ITEMS} sub-resource of the
     *            given resource
     */
    public Iterator<Resource> getItems(Resource resource) {
        return getItems(resource, ITEMS);
    }

    /**
     * Returns the child resources of the sub resource with the given name of
     * the given resource.
     * @param name
     *            The name of the resource
     * @param resource
     *            The resource
     * @return
     *            the child resources of the sub resource with the given name of
     *            the given resource.  
     */
    public Iterator<Resource> getItems(Resource resource, String name) {
        if (name == null || name.length() == 0) {
            name = ITEMS;
        }

        Resource items = resource.getChild(name);

        if (items != null) {
            return items.listChildren();
        }

        return Collections.<Resource> emptyList().iterator();
    }

    /**
     * Returns the child resource with the given name.
     * @param name
     *            The name of the resource
     * @return
     *            The child resource
     */
    public Resource getChild(String name) {
        return resource.getChild(name);
    }

    /**
     * Returns the parent of the resource. If the parent is of type
     * <code>{@link JcrConstants#NT_UNSTRUCTURED}</code> the direct parent is a
     * container (e.g. "items") and its parent is returned.
     * @return
     *            The parent of the resource
     */
    public Resource getParentResource() {
        if (parentResource != null) {
            return parentResource;
        }

        parentResource = resource.getParent();

        if (JcrConstants.NT_UNSTRUCTURED.equals(parentResource.getResourceType())) {
            parentResource = parentResource.getParent();
        }

        return parentResource;
    }
}
