/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.ui.components;

import aQute.bnd.annotation.ProviderType;
import javax.servlet.http.HttpServletRequest;

/**
 * Utility class for form field component.
 */
@ProviderType
public class Field {
    /**
     * The config of the component
     */
    private Config cfg;

    /**
     * The name of the CSS class to apply in order to hide empty field
     * components in default mode.
     */
    public static String HIDE_IN_DEFAULT_CLASS = "foundation-field-hide-in-default";

    /**
     * The suffix that specifies if a property is a mixed value (for bulk editing).
     */
    public static String IS_MIXED_SUFFIX = ".granite.mixed";

    /**
     * The request attribute to be set when doing bulk editing
     */
    private static final String BULK_EDIT_MODE_ATTRIBUTE = "granite.ui.field.bulkedit";

    public Field() { /* Empty on purpose: don't use it! */ }

    /**
    * Creates a new Field object with the given config.
    *
    * @param cfg
    *            The config of the component
    */
    public Field(Config cfg) {
        this.cfg = cfg;
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows
     * for example to hide empty field components in read only mode.
     *
     * @param isEmpty
     *            Indicate if the field is empty
     * @return
     *            the CSS class (or space separated class values)
     */
    public String getRootClass(boolean isEmpty) {
        return Field.getRootClass(this.cfg, isEmpty);
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows
     * for example to hide empty field components in read only mode.
     *
     * @param value
     *            The value of the field component
     * @return
     *            the CSS class (or space separated class values)
     */
    public String getRootClass(String value) {
        return Field.getRootClass(this.cfg, value);
    }

    /**
     * Returns <code>true</code> if the field is a mixed value; <code>false</code> otherwise.
     *
     * A field is a mixed value if it has a property named <code>cfg.get("name") + IS_MIXED_SUFFIX</code> which has the value of <code>true</code>.
     *
     * @param value
     *            The form value
     * @return
     *            True if the field component holds a mixed value, false otherwise
     */
    public boolean isMixed(Value value) {
        return Field.isMixed(this.cfg, value);
    }

    /**
     * Returns <code>true</code> if bulk edit is allowed on this field; <code>false</code> otherwise.
     *
     * A field could be bulk edited if it has a property named <code>allowBulkEdit</code> which has the value of <code>true</code>
     *
     * @return
     *            True if the field component could be bulk edited, false otherwise
     */
    public boolean isBulkEditAllowed() {
        return this.cfg.get("allowBulkEdit", false);
    }

    /**
     * Returns <code>true</code> if bulk edit mode is set; <code>false</code> otherwise
     *
     * Bulk edit mode is set if the request has an attribute <code>BULK_EDIT_MODE_ATTRIBUTE</code> which has the value of <code>true</code>
     *
     * @param request
     *            The request providing the parameter
     *
     * @return
     *            True if the bulk edit mode is set
     */
    public static boolean isBulkEditMode(HttpServletRequest request) {
        Boolean isBulkEditMode = (Boolean) request.getAttribute(BULK_EDIT_MODE_ATTRIBUTE);
        return isBulkEditMode == null ?  false : isBulkEditMode;
    }

    /**
     * Sets the bulk edit mode
     *
     * @param request
     *            The request that would hold the parameter
     */
    public static void setBulkEditMode(HttpServletRequest request) {
        request.setAttribute(BULK_EDIT_MODE_ATTRIBUTE, true);
    }

    /**
     * Clears the bulk edit mode
     *
     * @param request
     *            The request that was holding the parameter
     */
    public static void clearBulkEditMode(HttpServletRequest request) {
        request.removeAttribute(BULK_EDIT_MODE_ATTRIBUTE);
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows
     * for example to hide empty field components in read only mode.
     * 
     * @param cfg
     *            The config object of the field component
     * @param isEmpty
     *            Indicate if the field is empty
     * @return
     *            the CSS class (or space separated class values)
     */
    public static String getRootClass(Config cfg, boolean isEmpty) {
        String cls = "";

        // empty field is hidden in default mode if all of the following
        // conditions are true:
        // - value is empty
        // - config sets renderReadOnly = true (otherwise there is no rendering
        // for the default mode)
        // - config doesn't set showEmptyInReadOnly = true
        if (isEmpty && cfg.get("renderReadOnly", false) && !cfg.get("showEmptyInReadOnly", false)) {
            cls += HIDE_IN_DEFAULT_CLASS;
        }

        return cls;
    }

    /**
     * Returns the CSS class (or space separated class values) that have to be
     * applied to the root element of field component. The provided class allows
     * for example to hide empty field components in read only mode.
     * 
     * @param cfg
     *            The config object of the field component
     * @param value
     *            The value of the field component
     * @return
     *            the CSS class (or space separated class values)
     */
    public static String getRootClass(Config cfg, String value) {
        return getRootClass(cfg, value == null || "".equals(value));
    }

    /**
     * Returns <code>true</code> if the field is a mixed value; <code>false</code> otherwise.
     *
     * A field is a mixed value if it has a property named <code>cfg.get("name") + IS_MIXED_SUFFIX</code> which has the value of <code>true</code>.
     *
     * @param cfg
     *            The config object of the field component
     * @param value
     *            The form value
     * @return
     *            True if the field component holds a mixed value, false otherwise
     */
    public static boolean isMixed(Config cfg, Value value) {
        String name =  cfg.get("name") + IS_MIXED_SUFFIX;
        return value.getContentValue(name, false);
    }
}
