/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.ui.components;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.Cookie;

import org.apache.sling.api.SlingHttpServletRequest;

/**
 * A key-value map of client-side state. A client may set a state such that the
 * server can retrieve it to do something specific about this client based on
 * the state.
 */
public class State {
    private static final List<String> BLACKLIST = Arrays.asList("csrftoken", "login-token");
    
    private SlingHttpServletRequest request;
    private Set<String> names;

    public State(SlingHttpServletRequest request) {
        this.request = request;
    }

    /**
     * Returns the item value of the given name. Returns <code>null</code> if the
     * item is not found.
     * @param name the name of the item
     * @return the item value as a string
     */
    public String get(String name) {
        Item i = getItem(name);
        return i == null ? null : i.getString();
    }

    /**
     * Returns the item value of the given name. Returns the given defaultValue if
     * the item is not found.
     * @param name the name of the item
     * @param defaultValue the default value
     * @return the item value as a string or the default value if the item is not found
     */
    public String get(String name, String defaultValue) {
        Item i = getItem(name);
        return i == null ? defaultValue : i.getString();
    }

    /**
     * Returns the item value of the given name. Returns the given defaultValue if
     * the item is not found, or the value is not a boolean string ("true" or "false").
     * @param name the name of the item
     * @param defaultValue the default value
     * @return the item value as a boolean or the default value if the value is not a
     * boolean string
     */
    public boolean get(String name, boolean defaultValue) {
        Item i = getItem(name);
        
        if (i == null || (!"true".equals(i.getString()) && !"false".equals(i.getString()))) {
            return defaultValue;
        }

        return i.getBoolean();
    }

    /**
     * Returns the item value of the given name. Returns the given defaultValue if
     * the item is not found, or the value is not an integer string.
     * @param name the name of the item
     * @param defaultValue the default value
     * @return the item value as an int or the default value if the value is not an
     * int
     */
    public int get(String name, int defaultValue) {
        try {
            Item i = getItem(name);
            return i == null ? defaultValue : i.getInt();
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }
    
    /**
     * Returns the available names.
     * @return the available names 
     */
    public Iterator<String> names() {
        if (names == null) {
            names = new LinkedHashSet<String>();
            
            Cookie[] cookies = request.getCookies();
            
            if (cookies != null) {
                for (Cookie c : cookies) {
                    String name = c.getName();
                    
                    if (!BLACKLIST.contains(name.toLowerCase())) {
                        names.add(name);
                    }
                }
            }
        }
        
        return names.iterator();
    }
    
    /**
     * Returns the state item with the given name. Returns <code>null</code> if the
     * item is not found.
     * @param name the name of the item
     * @return the state item
     */
    public Item getItem(String name) {
        if (BLACKLIST.contains(name.toLowerCase())) {
            return null;
        }
        
        Cookie c = request.getCookie(name);

        if (c == null) {
            return null;
        }

        try {
            return new ItemImpl(name, URLDecoder.decode(c.getValue(), "utf-8"));
        } catch (UnsupportedEncodingException impossible) {
            throw new RuntimeException(impossible);
        }
    }
    
    /**
     * An item in the state.
     */
    public interface Item {
        /**
         * Returns the name of the item.
         * @return the name of the item
         */
        String getName();
        
        /**
         * Returns the value as string.
         * @return the value as string
         */
        String getString();
        
        /**
         * Returns the value as boolean.
         * The conversion is following {@link Boolean#parseBoolean(String)} semantic.
         * @return the value as boolean
         */
        boolean getBoolean();
        
        /**
         * Returns the value as int.
         * The conversion is following {@link Integer#parseInt(String)} semantic.
         * @return the value as int
         */
        int getInt() throws NumberFormatException;
    }
    
    private class ItemImpl implements Item {
        private String name;
        private String value;

        public ItemImpl(String name, String value) {
            this.name = name;
            this.value = value;
        }
        
        public String getName() {
            return name;
        }

        public String getString() {
            return value;
        }

        public boolean getBoolean() {
            return Boolean.parseBoolean(value);
        }

        public int getInt() throws NumberFormatException {
            return Integer.parseInt(value);
        }
    }
}
