/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.ui.components;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;

import com.adobe.granite.ui.components.FormData.NameNotFoundMode;

/**
 * A helper for the form field to access the {@link FormData}.
 *
 * <p>
 * To set the FormData, one of the following can be used (ordered by priority):
 * </p>
 * <ol>
 * <li>Set the request attribute using
 * {@link FormData#push(SlingHttpServletRequest, ValueMap, NameNotFoundMode)}</li>
 * <li>Set the request attribute using {@link #FORM_VALUESS_ATTRIBUTE} as the
 * key</li>
 * <li>Set the request attribute using {@link #CONTENTPATH_ATTRIBUTE} as the
 * key</li>
 * </ol>
 */
public class Value {
    /**
     * The Servlet attribute key to store the ValueMap of the form values.
     *
     * @deprecated Use
     *             {@link FormData#push(SlingHttpServletRequest, ValueMap, NameNotFoundMode)}
     *             instead
     */
    @Deprecated
    public static final String FORM_VALUESS_ATTRIBUTE = "granite.ui.form.values";

    /**
     * The Servlet attribute key to store the path to the resource holding the form
     * values.
     *
     * @deprecated Use
     *             {@link FormData#push(SlingHttpServletRequest, ValueMap, NameNotFoundMode)}
     *             instead
     */
    @Deprecated
    public static final String CONTENTPATH_ATTRIBUTE = "granite.ui.form.contentpath";

    @Nonnull
    private Config config;

    private FormData formData;

    /**
     * Instantiates a new Value.
     *
     * @param request
     *            The request storing the FormData
     * @param config
     *            The config of the form field component
     */
    public Value(@Nonnull SlingHttpServletRequest request, @Nonnull Config config) {
        this.config = config;

        formData = FormData.from(request);

        if (formData != null) {
            return;
        }

        ValueMap values = (ValueMap) request.getAttribute(FORM_VALUESS_ATTRIBUTE);

        if (values == null) {
            String contentPath = (String) request.getAttribute(CONTENTPATH_ATTRIBUTE);

            if (contentPath != null) {
                Resource contentResource = request.getResourceResolver().getResource(contentPath);

                if (contentResource != null) {
                    values = contentResource.getValueMap();
                }
            }
        }

        if (values != null) {
            formData = new FormData(values, NameNotFoundMode.IGNORE_FRESHNESS);
        }
    }

    /**
     * A shortcut to {@link #get(String, Object)}, with empty string as default
     * value.
     *
     * @param name
     *            The name of the field
     * @return The value converted to type T, or the value of {@code value}
     *         property, or the given default value, depending on the conditions
     *         described at {@link #get(String, Object)}.
     */
    @Nonnull
    public String get(@CheckForNull String name) {
        return get(name, "");
    }

    /**
     * Returns the value for the given name, converted to type T.
     *
     * <p>
     * In the {@code NameNotFoundMode#CHECK_FRESHNESS} mode, if the given name is
     * not found and the FormData is fresh, then the given fieldValue is returned.
     * Otherwise, the given defaultValue is returned.
     * </p>
     *
     * <p>
     * In the {@code NameNotFoundMode#IGNORE_FRESHNESS} mode, if the given name is
     * not found, then the given fieldValue is returned.
     * </p>
     *
     * <p>
     * If the FormData is {@code null} or {@code ignoreData} property of the config
     * is {@code true}, this method returns {@code value} property of the config.
     * </p>
     *
     * @param name
     *            The name of the field
     * @param defaultValue
     *            The default value
     * @param <T>
     *            The type of the value
     * @return The value converted to type T, or the value of {@code value}
     *         property, or the given default value, depending on the conditions
     *         described above.
     */
    @Nonnull
    public <T> T get(@CheckForNull String name, @Nonnull T defaultValue) {
        T fieldValue = config.get("value", defaultValue);

        if (formData == null || config.get("ignoreData", false)) {
            return fieldValue;
        }

        if (name != null) {
            return formData.get(name, fieldValue, defaultValue);
        } else {
            return fieldValue;
        }
    }

    /**
     * Returns the value for the given name, converted to type T.
     *
     * <p>
     * In the {@code NameNotFoundMode#CHECK_FRESHNESS} mode, if the given name is
     * not found and the FormData is fresh, then the given fieldValue is returned.
     * Otherwise, {@code null} is returned.
     * </p>
     *
     * <p>
     * In the {@code NameNotFoundMode#IGNORE_FRESHNESS} mode, if the given name is
     * not found, then the given fieldValue is returned.
     * </p>
     *
     * <p>
     * If the FormData is {@code null} or {@code ignoreData} property of the config
     * is {@code true}, this method returns {@code value} property of the config.
     * </p>
     *
     * @param name
     *            The name of the field
     * @param type
     *            The class of the type
     * @param <T>
     *            The type of the value
     * @return The value converted to type T, or the value of {@code value}
     *         property, or {@code null}, depending on the conditions described
     *         above.
     */
    @CheckForNull
    public <T> T get(@CheckForNull String name, @Nonnull Class<T> type) {
        T fieldValue = config.get("value", type);

        if (formData == null || config.get("ignoreData", false)) {
            return fieldValue;
        }

        if (name != null) {
            return formData.get(name, fieldValue, type);
        } else {
            return fieldValue;
        }
    }

    /**
     * A shortcut of {@link #val(String, Object)}, with name is taken from
     * {@code name} property of the config.
     *
     * @param fieldValue
     *            The value of the field
     * @param <T>
     *            The type of the value
     * @return The value converted to type T, or the given fieldValue, or
     *         {@code null}, depending on the conditions described at
     *         {@link #val(String, Object)}.
     */
    @CheckForNull
    public <T> T val(@Nonnull T fieldValue) {
        return val(config.get("name", String.class), fieldValue);
    }

    /**
     * Returns the value for the given name, converted to type T.
     *
     * <p>
     * In the {@code NameNotFoundMode#CHECK_FRESHNESS} mode, if the given name is
     * not found and the FormData is fresh, then the given fieldValue is returned.
     * Otherwise, {@code null} is returned.
     * </p>
     *
     * <p>
     * In the {@code NameNotFoundMode#IGNORE_FRESHNESS} mode, if the given name is
     * not found, then the given fieldValue is returned.
     * </p>
     *
     * <p>
     * If the FormData is {@code null} or {@code ignoreData} property of the config
     * is {@code true}, this method returns the given fieldValue.
     * </p>
     *
     * <p>
     * The key difference compared to {@link #get(String, Object)} is that this
     * method will return the given fieldValue instead of value from {@code value}
     * property of the config.
     * </p>
     *
     * @param name
     *            The name of the field
     * @param fieldValue
     *            The value of the field
     * @param <T>
     *            The type of the value
     * @return The value converted to type T, or the given fieldValue, or
     *         {@code null}, depending on the conditions described above.
     */
    @SuppressWarnings("unchecked")
    @CheckForNull
    public <T> T val(@CheckForNull String name, @Nonnull T fieldValue) {
        if (formData == null || config.get("ignoreData", false)) {
            return fieldValue;
        }

        if (name != null) {
            return formData.get(name, fieldValue, (Class<T>) fieldValue.getClass());
        } else {
            return fieldValue;
        }
    }

    /**
     * A shortcut of {@link #isSelected(String, String, boolean)}, with name is
     * taken from {@code name} property of the config.
     *
     * @param value
     *            The value of the field option to compare against
     * @param isFieldOptionSelected
     *            {@code true} if the field option is selected; {@code false}
     *            otherwise.
     * @return Whether the given value is selected or not, or the given
     *         isFieldOptionSelected, depending on the conditions described at
     *         {@link #isSelected(String, String, boolean)}.
     *
     */
    public boolean isSelected(@CheckForNull String value, boolean isFieldOptionSelected) {
        return isSelected(config.get("name", String.class), value, isFieldOptionSelected);
    }

    /**
     * Returns {@code true} if the given value of the field option is selected;
     * {@code false} otherwise.
     *
     * <p>
     * In the {@code NameNotFoundMode#CHECK_FRESHNESS} mode, if the given name is
     * not found and the FormData is fresh, then the given isFieldOptionSelected is
     * returned, {@code false} otherwise.
     * </p>
     *
     * <p>
     * In the {@code NameNotFoundMode#IGNORE_FRESHNESS} mode, if the given name is
     * not found, then the given isFieldOptionSelected is returned.
     * </p>
     *
     * <p>
     * If the FormData is {@code null} or {@code ignoreData} property of the config
     * is {@code true}, the given isFieldOptionSelected is returned.
     * </p>
     *
     * @param name
     *            The name of the field
     * @param value
     *            The value of the field option to compare against
     * @param isFieldOptionSelected
     *            {@code true} if the field option is selected; {@code false}
     *            otherwise.
     *
     * @return Whether the given value is selected or not, or the given
     *         isFieldOptionSelected, depending on the conditions described above.
     */
    public boolean isSelected(@CheckForNull String name, @CheckForNull String value, boolean isFieldOptionSelected) {
        if (formData == null || config.get("ignoreData", false)) {
            return isFieldOptionSelected;
        }

        if (name != null) {
            return formData.isSelected(name, value, isFieldOptionSelected);
        } else {
            return isFieldOptionSelected;
        }
    }

    /**
     * A shortcut to {@link #getContentValue(String, Object)}, with empty string as
     * default value.
     *
     * @param name
     *            The name of the field
     * @return The value converted to String, or an empty String if non existing or
     *         can't be converted.
     */
    @Nonnull
    public String getContentValue(@CheckForNull String name) {
        return getContentValue(name, "");
    }

    /**
     * Returns value from the given name, converted to the type of given default
     * value.
     *
     * @param name
     *            The name of the field
     * @param defaultValue
     *            The default value
     * @param <T>
     *            The type of the value
     * @return The value converted to type T, or the default value if non existing
     *         or can't be converted.
     */
    @Nonnull
    public <T> T getContentValue(@CheckForNull String name, @Nonnull T defaultValue) {
        if (formData == null || name == null) {
            return defaultValue;
        }

        return formData.getValueMap().get(name, defaultValue);
    }

    /**
     * Returns value from the given name, converted to the given type.
     *
     * @param name
     *            The name of the value to return
     * @param type
     *            The return type
     * @param <T>
     *            The type of the value
     * @return The named value converted to type T, or {@code null} if non existing
     *         or can't be converted.
     */
    @CheckForNull
    public <T> T getContentValue(@CheckForNull String name, @Nonnull Class<T> type) {
        if (formData == null || name == null) {
            return null;
        }

        return formData.getValueMap().get(name, type);
    }
}
